/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * libqmi-glib -- GLib/GIO based library to control QMI devices
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA.
 *
 * Copyright (C) 2012 Lanedo GmbH
 * Copyright (C) 2015 Velocloud Inc.
 * Copyright (C) 2012-2017 Aleksander Morgado <aleksander@aleksander.es>
 * Copyright (c) 2022 Qualcomm Innovation Center, Inc.
 */

#ifndef _LIBQMI_GLIB_QMI_ENUMS_WDS_H_
#define _LIBQMI_GLIB_QMI_ENUMS_WDS_H_

#if !defined (__LIBQMI_GLIB_H_INSIDE__) && !defined (LIBQMI_GLIB_COMPILATION)
#error "Only <libqmi-glib.h> can be included directly."
#endif

/**
 * SECTION: qmi-enums-wds
 * @title: WDS enumerations and flags
 * @short_description: Enumerations and flags in the WDS service.
 *
 * This section defines enumerations and flags used in the WDS service
 * interface.
 */

#include <glib.h>

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Start Network' message */

/**
 * QmiWdsIpFamily:
 * @QMI_WDS_IP_FAMILY_UNKNOWN: unknown. Since 1.18.
 * @QMI_WDS_IP_FAMILY_IPV4: IPv4.
 * @QMI_WDS_IP_FAMILY_IPV6: IPv6.
 * @QMI_WDS_IP_FAMILY_UNSPECIFIED: None specified.
 *
 * Type of IP family.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_IP_FAMILY_UNKNOWN     = 0,
    QMI_WDS_IP_FAMILY_IPV4        = 4,
    QMI_WDS_IP_FAMILY_IPV6        = 6,
    QMI_WDS_IP_FAMILY_UNSPECIFIED = 8
} QmiWdsIpFamily;

/**
 * QmiWdsTechnologyPreference:
 * @QMI_WDS_TECHNOLOGY_PREFERENCE_ALLOW_3GPP: 3GPP allowed.
 * @QMI_WDS_TECHNOLOGY_PREFERENCE_ALLOW_3GPP2: 3GPP2 allowed.
 *
 * Type of network allowed when trying to connect.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_TECHNOLOGY_PREFERENCE_ALLOW_3GPP  = 1 << 0,
    QMI_WDS_TECHNOLOGY_PREFERENCE_ALLOW_3GPP2 = 1 << 1
} QmiWdsTechnologyPreference;

/**
 * QmiWdsExtendedTechnologyPreference:
 * @QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_CDMA: Use CDMA.
 * @QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_UMTS: Use UMTS.
 * @QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_EPC: Use EPC, LTE.
 * @QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_EMBMS: Use eMBMS.
 * @QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_MODEM_LINK_LOCAL: Use modem link-local.
 *
 * Type of network allowed when trying to connect.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_CDMA             = 32769,
    QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_UMTS             = 32772,
    QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_EPC              = 34944,
    QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_EMBMS            = 34946,
    QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_MODEM_LINK_LOCAL = 34952,
} QmiWdsExtendedTechnologyPreference;

/**
 * QmiWdsCallType:
 * @QMI_WDS_CALL_TYPE_LAPTOP: Laptop call.
 * @QMI_WDS_CALL_TYPE_EMBEDDED: Embedded call.
 *
 * Type of call to originate.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_CALL_TYPE_LAPTOP   = 0,
    QMI_WDS_CALL_TYPE_EMBEDDED = 1
} QmiWdsCallType;

/**
 * QmiWdsCallEndReason:
 * @QMI_WDS_CALL_END_REASON_GENERIC_UNSPECIFIED: Unspecified reason.
 * @QMI_WDS_CALL_END_REASON_GENERIC_CLIENT_END: Client end.
 * @QMI_WDS_CALL_END_REASON_GENERIC_NO_SERVICE: No service.
 * @QMI_WDS_CALL_END_REASON_GENERIC_FADE: Fade.
 * @QMI_WDS_CALL_END_REASON_GENERIC_RELEASE_NORMAL: Release normal.
 * @QMI_WDS_CALL_END_REASON_GENERIC_ACCESS_ATTEMPT_IN_PROGRESS: Access attempt in progress.
 * @QMI_WDS_CALL_END_REASON_GENERIC_ACCESS_FAILURE: Access Failure.
 * @QMI_WDS_CALL_END_REASON_GENERIC_REDIRECTION_OR_HANDOFF: Redirection or handoff.
 * @QMI_WDS_CALL_END_REASON_GENERIC_CLOSE_IN_PROGRESS: Close in progress.
 * @QMI_WDS_CALL_END_REASON_GENERIC_AUTHENTICATION_FAILED: Authentication failed.
 * @QMI_WDS_CALL_END_REASON_GENERIC_INTERNAL_ERROR: Internal error.
 * @QMI_WDS_CALL_END_REASON_CDMA_LOCK: CDMA specific. Phone is CDMA-locked.
 * @QMI_WDS_CALL_END_REASON_CDMA_INTERCEPT: CDMA specific. Received intercept from the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_REORDER: CDMA specific. Received reorder from the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_RELEASE_SO_REJECT: CDMA specific. Received release from the BS, SO reject.
 * @QMI_WDS_CALL_END_REASON_CDMA_INCOMING_CALL: CDMA specific. Received incoming call from the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_ALERT_STOP: CDMA specific. Received alert stop from the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_ACTIVATION: CDMA specific. Received end activation.
 * @QMI_WDS_CALL_END_REASON_CDMA_MAX_ACCESS_PROBES: CDMA specific. Maximum access probes transmitted.
 * @QMI_WDS_CALL_END_REASON_CDMA_CCS_NOT_SUPPORTED_BY_BS: CDMA specific. Concurrent service not supported by the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_NO_RESPONSE_FROM_BS: CDMA specific. No response received from the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_REJECTED_BY_BS: CDMA specific. Rejected by the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_INCOMPATIBLE: CDMA specific. Concurrent services requested are incompatible.
 * @QMI_WDS_CALL_END_REASON_CDMA_ALREADY_IN_TC: CDMA specific. Already in TC.
 * @QMI_WDS_CALL_END_REASON_CDMA_USER_CALL_ORIGINATED_DURING_GPS: CDMA specific. Call originated during GPS.
 * @QMI_WDS_CALL_END_REASON_CDMA_USER_CALL_ORIGINATED_DURING_SMS: CDMA specific. Call originated during SMS.
 * @QMI_WDS_CALL_END_REASON_CDMA_NO_SERVICE: CDMA specific. No service.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_CONFERENCE_FAILED: GSM/WCDMA specific. Call origination request failed.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_INCOMING_REJECTED: GSM/WCDMA specific. Client rejected incoming call.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_NO_SERVICE: GSM/WCDMA specific. No service.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_NETWORK_END: GSM/WCDMA specific. Network ended the call.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_LLC_SNDCP_FAILURE: GSM/WCDMA specific. LLC or SNDCP failure.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_INSUFFICIENT_RESOURCES: GSM/WCDMA specific. Insufficient resources.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_TEMPORARILY_OUT_OF_ORDER: GSM/WCDMA specific. Service option temporarily out of order.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_NSAPI_ALREADY_USED: GSM/WCDMA specific. NSAPI already used.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_REGULAR_DEACTIVATION: GSM/WCDMA specific. Regular PDP context deactivation.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_NETWORK_FAILURE: GSM/WCDMA specific. Network failure.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_REATTACH_REQUIRED: GSM/WCDMA specific. Reattach required.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_PROTOCOL_ERROR: GSM/WCDMA specific. Protocol error.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPERATOR_DETERMINED_BARRING: GSM/WCDMA specific. Operator-determined barring.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_APN: GSM/WCDMA specific. Unknown or missing APN.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_PDP: GSM/WCDMA specific. Unknown PDP address or type.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_GGSN_REJECT: GSM/WCDMA specific. Activation rejected by GGSN.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_ACTIVATION_REJECT: GSM/WCDMA specific. Activation rejected.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_NOT_SUPPORTED: GSM/WCDMA specific. Service option not supported.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_UNSUBSCRIBED: GSM/WCDMA specific. Service option not subscribed.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_QOS_NOT_ACCEPTED: GSM/WCDMA specific. QoS not accepted.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_TFT_SEMANTIC_ERROR: GSM/WCDMA specific. Semantic error in TFT operation.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_TFT_SYNTAX_ERROR: GSM/WCDMA specific. Syntax error in TFT operation.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_PDP_CONTEXT: GSM/WCDMA specific. Unknown PDP context.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_FILTER_SEMANTIC_ERROR: GSM/WCDMA specific. Semantic error in packet filters.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_FILTER_SYNTAX_ERROR: GSM/WCDMA specific. Syntax error in packet filters.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_PDP_WITHOUT_ACTIVE_TFT: GSM/WCDMA specific. PDP context without TFT activated.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_INVALID_TRANSACTION_ID: GSM/WCDMA specific. Invalid transaction ID.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_INCORRECT_SEMANTIC: GSM/WCDMA specific. Message incorrect semantically.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_INVALID_MANDATORY_INFO: GSM/WCDMA specific. Invalid mandatory information.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_TYPE_UNSUPPORTED: GSM/WCDMA specific. Message type not implemented.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_TYPE_NONCOMPATIBLE_STATE: GSM/WCDMA specific. Message not compatible with state.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_INFO_ELEMENT: GSM/WCDMA specific. Information element unknown.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_CONDITIONAL_IE_ERROR: GSM/WCDMA specific. Conditional IE error.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_AND_PROTOCOL_STATE_UNCOMPATIBLE: GSM/WCDMA specific. Message and protocol state uncompatible.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_APN_TYPE_CONFLICT: GSM/WCDMA specific. APN type conflict.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_NO_GPRS_CONTEXT: GSM/WCDMA specific. No GPRS context.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_FEATURE_NOT_SUPPORTED: GSM/WCDMA specific. Feature not supported.
 * @QMI_WDS_CALL_END_REASON_EVDO_CONNECTION_DENY_GENERAL_OR_BUSY: CDMA/EVDO specific. Received Connection Deny: General or Network busy.
 * @QMI_WDS_CALL_END_REASON_EVDO_CONNECTION_DENY_BILLING_OR_AUTHENTICATION_FAILURE: CDMA/EVDO specific. Received Connection Deny: Billing or Authentication failure.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_CHANGE: CDMA/EVDO specific. Change HDR.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_EXIT: CDMA/EVDO specific. Exit HDR.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_NO_SESSION: CDMA/EVDO specific. No HDR session.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_ORIGINATION_DURING_GPS_FIX: CDMA/EVDO specific. HDR call ended in favor of a GPS fix.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_CONNECTION_SETUP_TIMEOUT: CDMA/EVDO specific. Connection setup timeout.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_RELEASED_BY_CM: CDMA/EVDO specific. Released HDR call by call manager.
 *
 * Reason for ending the call.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    /* Generic reasons */
    QMI_WDS_CALL_END_REASON_GENERIC_UNSPECIFIED                = 1,
    QMI_WDS_CALL_END_REASON_GENERIC_CLIENT_END                 = 2,
    QMI_WDS_CALL_END_REASON_GENERIC_NO_SERVICE                 = 3,
    QMI_WDS_CALL_END_REASON_GENERIC_FADE                       = 4,
    QMI_WDS_CALL_END_REASON_GENERIC_RELEASE_NORMAL             = 5,
    QMI_WDS_CALL_END_REASON_GENERIC_ACCESS_ATTEMPT_IN_PROGRESS = 6,
    QMI_WDS_CALL_END_REASON_GENERIC_ACCESS_FAILURE             = 7,
    QMI_WDS_CALL_END_REASON_GENERIC_REDIRECTION_OR_HANDOFF     = 8,
    QMI_WDS_CALL_END_REASON_GENERIC_CLOSE_IN_PROGRESS          = 9,
    QMI_WDS_CALL_END_REASON_GENERIC_AUTHENTICATION_FAILED      = 10,
    QMI_WDS_CALL_END_REASON_GENERIC_INTERNAL_ERROR             = 11,

    /* CDMA specific reasons */
    QMI_WDS_CALL_END_REASON_CDMA_LOCK                            = 500,
    QMI_WDS_CALL_END_REASON_CDMA_INTERCEPT                       = 501,
    QMI_WDS_CALL_END_REASON_CDMA_REORDER                         = 502,
    QMI_WDS_CALL_END_REASON_CDMA_RELEASE_SO_REJECT               = 503,
    QMI_WDS_CALL_END_REASON_CDMA_INCOMING_CALL                   = 504,
    QMI_WDS_CALL_END_REASON_CDMA_ALERT_STOP                      = 505,
    QMI_WDS_CALL_END_REASON_CDMA_ACTIVATION                      = 506,
    QMI_WDS_CALL_END_REASON_CDMA_MAX_ACCESS_PROBES               = 507,
    QMI_WDS_CALL_END_REASON_CDMA_CCS_NOT_SUPPORTED_BY_BS         = 508,
    QMI_WDS_CALL_END_REASON_CDMA_NO_RESPONSE_FROM_BS             = 509,
    QMI_WDS_CALL_END_REASON_CDMA_REJECTED_BY_BS                  = 510,
    QMI_WDS_CALL_END_REASON_CDMA_INCOMPATIBLE                    = 511,
    QMI_WDS_CALL_END_REASON_CDMA_ALREADY_IN_TC                   = 512,
    QMI_WDS_CALL_END_REASON_CDMA_USER_CALL_ORIGINATED_DURING_GPS = 513,
    QMI_WDS_CALL_END_REASON_CDMA_USER_CALL_ORIGINATED_DURING_SMS = 514,
    QMI_WDS_CALL_END_REASON_CDMA_NO_SERVICE                      = 515,

    /* GSM/WCDMA specific reasons */
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_CONFERENCE_FAILED                       = 1000,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_INCOMING_REJECTED                       = 1001,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_NO_SERVICE                              = 1002,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_NETWORK_END                             = 1003,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_LLC_SNDCP_FAILURE                       = 1004,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_INSUFFICIENT_RESOURCES                  = 1005,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_TEMPORARILY_OUT_OF_ORDER         = 1006,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_NSAPI_ALREADY_USED                      = 1007,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_REGULAR_DEACTIVATION                    = 1008,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_NETWORK_FAILURE                         = 1009,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_REATTACH_REQUIRED                       = 1010,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_PROTOCOL_ERROR                          = 1011,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPERATOR_DETERMINED_BARRING             = 1012,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_APN                             = 1013,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_PDP                             = 1014,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_GGSN_REJECT                             = 1015,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_ACTIVATION_REJECT                       = 1016,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_NOT_SUPPORTED                    = 1017,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_UNSUBSCRIBED                     = 1018,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_QOS_NOT_ACCEPTED                        = 1019,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_TFT_SEMANTIC_ERROR                      = 1020,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_TFT_SYNTAX_ERROR                        = 1021,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_PDP_CONTEXT                     = 1022,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_FILTER_SEMANTIC_ERROR                   = 1023,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_FILTER_SYNTAX_ERROR                     = 1024,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_PDP_WITHOUT_ACTIVE_TFT                  = 1025,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_INVALID_TRANSACTION_ID                  = 1026,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_INCORRECT_SEMANTIC              = 1027,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_INVALID_MANDATORY_INFO                  = 1028,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_TYPE_UNSUPPORTED                = 1029,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_TYPE_NONCOMPATIBLE_STATE        = 1030,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_INFO_ELEMENT                    = 1031,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_CONDITIONAL_IE_ERROR                    = 1032,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_AND_PROTOCOL_STATE_UNCOMPATIBLE = 1033,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_APN_TYPE_CONFLICT                       = 1034,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_NO_GPRS_CONTEXT                         = 1035,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_FEATURE_NOT_SUPPORTED                   = 1036,

    /* EVDO specific reasons */
    QMI_WDS_CALL_END_REASON_EVDO_CONNECTION_DENY_GENERAL_OR_BUSY                   = 1500,
    QMI_WDS_CALL_END_REASON_EVDO_CONNECTION_DENY_BILLING_OR_AUTHENTICATION_FAILURE = 1501,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_CHANGE                                        = 1502,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_EXIT                                          = 1503,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_NO_SESSION                                    = 1504,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_ORIGINATION_DURING_GPS_FIX                    = 1505,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_CONNECTION_SETUP_TIMEOUT                      = 1506,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_RELEASED_BY_CM                                = 1507
} QmiWdsCallEndReason;

/**
 * QmiWdsVerboseCallEndReasonType:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_MIP: Mobile IP.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_INTERNAL: Internal.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_CM: Call manager.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_3GPP: 3GPP.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_PPP: PPP.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_EHRPD: eHRPD.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_IPV6: IPv6.
 *
 * Type of verbose call end reason.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_MIP      = 1,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_INTERNAL = 2,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_CM       = 3,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_3GPP     = 6,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_PPP      = 7,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_EHRPD    = 8,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_IPV6     = 9
} QmiWdsVerboseCallEndReasonType;

/**
 * qmi_wds_verbose_call_end_reason_get_string:
 * @type: a #QmiWdsVerboseCallEndReasonType.
 * @reason: the reason id.
 *
 * Get verbose call end reason based on the reason type.
 *
 * Returns: (transfer none): a string with the reason, or %NULL if not found. Do not free the returned value.
 *
 * Since: 1.0
 */
const gchar *qmi_wds_verbose_call_end_reason_get_string (QmiWdsVerboseCallEndReasonType type,
                                                         gint16                         reason);

/**
 * QmiWdsVerboseCallEndReasonMip:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_ERROR_REASON_UNKNOWN: Unknown reason.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REASON_UNSPECIFIED: FA error. Reason unspecified.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_ADMINISTRATIVELY_PROHIBITED: FA error. Administratively prohibited.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_INSUFFICIENT_RESOURCES: FA error. Insufficient resources.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MOBILE_NODE_AUTHENTICATION_FAILURE: FA error. Mobile node authenticatin failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_HA_AUTHENTICATION_FAILURE: FA error. HA authentication failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REQUESTED_LIFETIME_TOO_LONG: FA error. Requested lifetime too long.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MALFORMED_REQUEST: FA error. Malformed request.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MALFORMED_REPLY: FA error. Malformed reply.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_ENCAPSULATION_UNAVAILABLE: FA error. Encapsulation unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_VJHC_UNAVAILABLE: FA error. VJHC unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REVERSE_TUNNEL_UNAVAILABLE: FA error. Reverse tunnel unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REVERSE_TUNNEL_MANDATORY_AND_T_BIT_NOT_SET: FA error. Reverse tunnel mandatory and T bit not set.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_DELIVERY_STYLE_NOT_SUPPORTED: FA error. Delivery style not supported.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_NAI: FA error. Missing NAI.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_HA: FA error. Missing HA.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_HOME_ADDRESS: FA error. Missing home address.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_UNKNOWN_CHALLENGE: FA error. Unknown challenge.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_CHALLENGE: FA error. Missing challenge.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_STALE_CHALLENGE: FA error. Stale challenge.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REASON_UNSPECIFIED: HA error. Reason unspecified.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_ADMINISTRATIVELY_PROHIBITED: HA error. Administratively prohibited.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_INSUFFICIENT_RESOURCES: HA error. Insufficient resources.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_MOBILE_NODE_AUTHENTICATION_FAILURE: HA error. Mobile node authentication failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_FA_AUTHENTICATION_FAILURE: HA error. FA authentication failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REGISTRATION_ID_MISMATCH: HA error. Registration ID mismatch.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_MALFORMED_REQUEST: HA error. Malformed request.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_UNKNOWN_HA_ADDRESS: HA error. Unknown HA address.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REVERSE_TUNNEL_UNAVAILABLE: HA error. Reverse tunnel unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REVERSE_TUNNEL_MANDATORY_AND_T_BIT_NOT_SET: HA error. Reverse tunnel mandatory and T bit not set.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_ENCAPSULATION_UNAVAILABLE: HA error. Encapsulation unavailable.
 *
 * Mobile IP specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_MIP.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_ERROR_REASON_UNKNOWN                                = -1,

    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REASON_UNSPECIFIED                         = 64,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_ADMINISTRATIVELY_PROHIBITED                = 65,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_INSUFFICIENT_RESOURCES                     = 66,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MOBILE_NODE_AUTHENTICATION_FAILURE         = 67,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_HA_AUTHENTICATION_FAILURE                  = 68,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REQUESTED_LIFETIME_TOO_LONG                = 69,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MALFORMED_REQUEST                          = 70,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MALFORMED_REPLY                            = 71,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_ENCAPSULATION_UNAVAILABLE                  = 72,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_VJHC_UNAVAILABLE                           = 73,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REVERSE_TUNNEL_UNAVAILABLE                 = 74,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REVERSE_TUNNEL_MANDATORY_AND_T_BIT_NOT_SET = 75,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_DELIVERY_STYLE_NOT_SUPPORTED               = 79,

    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_NAI                                = 97,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_HA                                 = 98,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_HOME_ADDRESS                       = 99,

    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_UNKNOWN_CHALLENGE                          = 104,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_CHALLENGE                          = 105,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_STALE_CHALLENGE                            = 106,

    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REASON_UNSPECIFIED                         = 128,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_ADMINISTRATIVELY_PROHIBITED                = 129,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_INSUFFICIENT_RESOURCES                     = 130,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_MOBILE_NODE_AUTHENTICATION_FAILURE         = 131,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_FA_AUTHENTICATION_FAILURE                  = 132,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REGISTRATION_ID_MISMATCH                   = 133,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_MALFORMED_REQUEST                          = 134,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_UNKNOWN_HA_ADDRESS                         = 136,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REVERSE_TUNNEL_UNAVAILABLE                 = 137,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REVERSE_TUNNEL_MANDATORY_AND_T_BIT_NOT_SET = 138,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_ENCAPSULATION_UNAVAILABLE                  = 139
} QmiWdsVerboseCallEndReasonMip;

/**
 * QmiWdsVerboseCallEndReasonInternal:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR: Internal error.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CALL_ENDED: Call ended.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNKNOWN_INTERNAL_CAUSE: Unknown internal cause.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNKNOWN_CAUSE: Unknown cause.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CLOSE_IN_PROGRESS: Close in progress.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_NETWORK_INITIATED_TERMINATION: Network initiated termination.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APP_PREEMPTED: App preempted.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV4_CALL_DISALLOWED: PDN IPv4 call disallowed. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV4_CALL_THROTTLED: PDN IPv4 call throttled. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV6_CALL_DISALLOWED: PDN IPv6 call disallowed. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV6_CALL_THROTTLED: PDN IPv6 call throttled. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MODEM_RESTART: Modem restart. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDP_PPP_NOT_SUPPORTED: PDP PPP not supported. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNPREFERRED_RAT: Unpreferred RAT. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PHYSICAL_LINK_CLOSE_IN_PROGRESS: Physical link close in progress. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_PENDING_HANDOVER: APN pending handover. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PROFILE_BEARER_INCOMPATIBLE: Profile bearer incompatible. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MMGDSI_CARD_EVENT: MMGDSI Card event. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_LPM_OR_POWER_DOWN: LPM or power down. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_DISABLED: APN disabled. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MPIT_EXPIRED: MPIT expired. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_IPV6_ADDRESS_TRANSFER_FAILED: IPv6 address transfer failed. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_TRAT_SWAP_FAILED: TRAT swap failed. Since 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_EHRPD_TO_HRPD_FALLBACK: eHRPD to HRPD fallback. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MANDATORY_APN_DISABLED: Mandatory APN disabled. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MIP_CONFIG_FAILURE: MIP config failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_INACTIVITY_TIMER_EXPIRED: PDN inactivity timeout expired. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MAX_V4_CONNECTIONS: Max IPv4 connections. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MAX_V6_CONNECTIONS: Max IPv6 connections. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_MISMATCH: APN mismatch. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_IP_VERSION_MISMATCH: IP version mismatch. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_DUN_CALL_DISALLOWED: DUN call disallowed. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INVALID_PROFILE: Invalid profile. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_EPC_NONEPC_TRANSITION: EPC to non-EPC transition. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INVALID_PROFILE_ID: Invalid profile ID. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CALL_ALREADY_PRESENT: Call already present. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INTERFACE_IN_USE: Interface in use. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_IP_PDP_MISMATCH: IP PDP mismatch. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_DISALLOWED_ON_ROAMING: APN disallowed on roaming. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_PARAMETER_CHANGE: APN parameter change. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INTERFACE_IN_USE_CONFIG_MATCH: Interface in use config match. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_NULL_APN_DISALLOWED: NULL APN disallowed. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_THERMAL_MITIGATION: Thermal mitigation. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_SUBS_ID_MISMATCH: Subs ID mismatch Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_DATA_SETTINGS_DISABLED: Data settings disabled. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_DATA_ROAMING_SETTINGS_DISABLED: Data roaming settings disabled. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_FORMAT_INVALID: APN format invalid. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_DDS_CALL_ABORT: DDS call abort. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_VALIDATION_FAILURE: Validation failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PROFILES_NOT_COMPATIBLE: Profiles not compatible. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_NULL_RESOLVED_APN_NO_MATCH: Null resolved APN no match. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INVALID_APN_NAME: Invalid APN name. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_DDS_SWITCH_IN_PROGRESS: Dedicated Data Subscription (DDS) switch in progress. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CALL_DISALLOWED_IN_ROAMING: Call disallowed in roaming. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MO_EXCEPTIONAL_NOT_SUPPORTED: MO Exceptional not supported. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_NON_IP_NOT_SUPPORTED: Non-IP not supported. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR_PDN_NON_IP_CALL_THROTTLED: Error PDN non-IP call throttled. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR_PDN_NON_IP_CALL_DISALLOWED: Error PDN non-IP call disallowed. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR_NON_IP_TYPE_MISMATCH: Error non-IP type mismatch. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR_MAX_NB_PDN_REACHED: Error maximum NB PDN reached. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INVALID_APN: Invalid APN. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_SLICE_NOT_ALLOWED: Slice not allowed. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ROUTING_FAIL: Routing failure. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ROUTING_CHANGED: Routing changed. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_LOCAL_AREA_DATA_NETWORK_DATA_NETWORK_NAME_NOT_AVAILABLE: Local Area Data Network (LADN) Data Network Name (DNN) not allowed. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_TYPE_MISMATCH: APN type mismatch. Since 1.36.
 *
 * Internal call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_INTERNAL.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR                           = 201,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CALL_ENDED                      = 202,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNKNOWN_INTERNAL_CAUSE          = 203,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNKNOWN_CAUSE                   = 204,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CLOSE_IN_PROGRESS               = 205,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_NETWORK_INITIATED_TERMINATION   = 206,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APP_PREEMPTED                   = 207,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV4_CALL_DISALLOWED        = 208,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV4_CALL_THROTTLED         = 209,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV6_CALL_DISALLOWED        = 210,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV6_CALL_THROTTLED         = 211,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MODEM_RESTART                   = 212,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDP_PPP_NOT_SUPPORTED           = 213,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNPREFERRED_RAT                 = 214,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PHYSICAL_LINK_CLOSE_IN_PROGRESS = 215,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_PENDING_HANDOVER            = 216,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PROFILE_BEARER_INCOMPATIBLE     = 217,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MMGDSI_CARD_EVENT               = 218,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_LPM_OR_POWER_DOWN               = 219,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_DISABLED                    = 220,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MPIT_EXPIRED                    = 221,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_IPV6_ADDRESS_TRANSFER_FAILED    = 222,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_TRAT_SWAP_FAILED                = 223,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_EHRPD_TO_HRPD_FALLBACK          = 224,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MANDATORY_APN_DISABLED          = 225,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MIP_CONFIG_FAILURE              = 226,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_INACTIVITY_TIMER_EXPIRED    = 227,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MAX_V4_CONNECTIONS              = 228,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MAX_V6_CONNECTIONS              = 229,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_MISMATCH                    = 230,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_IP_VERSION_MISMATCH             = 231,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_DUN_CALL_DISALLOWED             = 232,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INVALID_PROFILE                 = 233,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_EPC_NONEPC_TRANSITION           = 234,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INVALID_PROFILE_ID              = 235,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CALL_ALREADY_PRESENT            = 236,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INTERFACE_IN_USE                = 237,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_IP_PDP_MISMATCH                 = 238,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_DISALLOWED_ON_ROAMING       = 239,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_PARAMETER_CHANGE            = 240,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INTERFACE_IN_USE_CONFIG_MATCH   = 241,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_NULL_APN_DISALLOWED             = 242,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_THERMAL_MITIGATION              = 243,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_SUBS_ID_MISMATCH                = 244,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_DATA_SETTINGS_DISABLED          = 245,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_DATA_ROAMING_SETTINGS_DISABLED  = 246,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_FORMAT_INVALID              = 247,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_DDS_CALL_ABORT                  = 248,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_VALIDATION_FAILURE              = 249,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PROFILES_NOT_COMPATIBLE         = 251,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_NULL_RESOLVED_APN_NO_MATCH      = 252,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INVALID_APN_NAME                = 253,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_DDS_SWITCH_IN_PROGRESS          = 254,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CALL_DISALLOWED_IN_ROAMING      = 255,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MO_EXCEPTIONAL_NOT_SUPPORTED    = 256,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_NON_IP_NOT_SUPPORTED            = 257,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR_PDN_NON_IP_CALL_THROTTLED  = 258,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR_PDN_NON_IP_CALL_DISALLOWED = 259,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR_NON_IP_TYPE_MISMATCH       = 261,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR_MAX_NB_PDN_REACHED         = 262,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_INVALID_APN                      = 263,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_SLICE_NOT_ALLOWED                = 264,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ROUTING_FAIL                     = 265,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ROUTING_CHANGED                  = 266,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_LOCAL_AREA_DATA_NETWORK_DATA_NETWORK_NAME_NOT_AVAILABLE = 267,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_TYPE_MISMATCH                = 268,
} QmiWdsVerboseCallEndReasonInternal;

/**
 * QmiWdsVerboseCallEndReasonCm:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CDMA_LOCK: CDMA specific. Phone is CDMA-locked.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INTERCEPT: CDMA specific. Received intercept from the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_REORDER: CDMA specific. Received reorder from the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RELEASE_SO_REJECT: CDMA specific. Received release from the BS, SO reject.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMING_CALL: CDMA specific. Received incoming call from the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ALERT_STOP: CDMA specific. Received alert stop from the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACTIVATION: CDMA specific. Received end activation.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MAX_ACCESS_PROBES: CDMA specific. Maximum access probes transmitted.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CCS_NOT_SUPPORTED_BY_BS: CDMA specific. Concurrent service not supported by the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_RESPONSE_FROM_BS: CDMA specific. No response received from the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_REJECTED_BY_BS: CDMA specific. Rejected by the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMPATIBLE: CDMA specific. Concurrent services requested are incompatible.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ALREADY_IN_TC: CDMA specific. Already in TC.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_USER_CALL_ORIGINATED_DURING_GPS: CDMA specific. Call originated during GPS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_USER_CALL_ORIGINATED_DURING_SMS: CDMA specific. Call originated during SMS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_CDMA_SERVICE: CDMA specific. No service.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MC_ABORT: MC abort. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PSIST_NG: Persistence test failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_UIM_NOT_PRESENT: UIM not present. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RETRY_ORDER: Retry order.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_BLOCK: Access blocked. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_BLOCK_ALL: Access blocked for all. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_IS707B_MAX_ACCESS_PROBES: Maximum access probes for IS-707B call. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_THERMAL_EMERGENCY: Thermal emergency. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CALL_ORIGINATION_THROTTLED: Call origination throttled. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_USER_CALL_ORIGINATED: Voice call originated and concurrent voice/data not supported. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONFERENCE_FAILED: Conference failed. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMING_REJECTED: Incoming rejected.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_GATEWAY_SERVICE: No gateway service.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_GPRS_CONTEXT: No GPRS context.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ILLEGAL_MS: Illegal MS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ILLEGAL_ME: Illegal ME.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_AND_NON_GPRS_SERVICES_NOT_ALLOWED: GPRS and non GPRS services not allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_SERVICES_NOT_ALLOWED: GPRS services not allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MS_IDENTITY_NOT_DERIVED_BY_THE_NETWORK: MS identity not derived by the network.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_IMPLICITLY_DETACHED: Implicitly detached.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PLMN_NOT_ALLOWED: PLMN not allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LA_NOT_ALLOWED: LA not allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_SERVICES_NOT_ALLOWED_IN_PLMN: GPRS services not allowed in PLMN.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_DUPLICATE: PDP duplicate.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_UE_RAT_CHANGE: UE radio access technology change.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONGESTION: Congestion.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_PDP_CONTEXT_ACTIVATED: No PDP context activated.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_CLASS_DSAC_REJECTION: Access class DSAC rejection.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_ACTIVATE_MAX_RETRY_FAILED: Maximum retries failed to activate PDP context. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RAB_FAILURE: RAB failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EPS_SERVICE_NOT_ALLOWED: EPS service not allowed. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_TRACKING_AREA_NOT_ALLOWED: Tracking area not allowed. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ROAMING_NOT_ALLOWED_IN_TRACKING_AREA: Roaming not allowed in tracking area. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_SUITABLE_CELLS_IN_TRACKING_AREA: No suitable cells in tracking area. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NOT_AUTHORIZED_CLOSED_SUBSCRIBER_GROUP: Not authorized in closed subscriber group. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_UNKNOWN_EPS_BEARER_CONTEXT: EPS Session Management, unknown EPS bearer context. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_DRB_RELEASED_AT_RRC: DRB released at RRC. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NAS_SIGNAL_CONNECTION_RELEASED: NAS signal connection released. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_DETACHED: EPS Mobility Management, detached. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_ATTACH_FAILED: EPS Mobility Management, attach failed. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_ATTACH_STARTED: EPS Mobility Management, attach started. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LTE_NAS_SERVICE_REQUEST_FAILED: LTE NAS service request failed. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_ACTIVE_DEDICATED_BEARER_REACTIVATED_BY_NW: EPS Session Management, active dedicated bearer reactivated by network. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_LOWER_LAYER_FAILURE: EPS Session Management, lower layer failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_SYNC_UP_WITH_NW: EPS Session Management, sync up with network. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_NW_ACTIVATED_DEDICATED_BEARER_WITH_ID_OF_DEFAULT_BEARER: EPS Session Management, network activated dedicated bearer with id of default bearer. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_BAD_OTA_MESSAGE: EPS Session Management, bad OTA message. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_DS_REJECTED_CALL: EPS Session Management, DS rejected call. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_CONTEXT_TRANSFERRED_DUE_TO_IRAT: EPS Session Management, context transferred due to IRAT. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_DS_EXPLICIT_DEACT: DS explicit deactivation. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_LOCAL_CAUSE_NONE: EPS Session Management, local cause none. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LTE_NAS_SERVICE_REQUEST_FAILED_NO_THROTTLE: LTE NAS service request failed, no throttle. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACL_FAILURE: ACL failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LTE_NAS_SERVICE_REQUEST_FAILED_DS_DISALLOW: LTE NAS service request failed, DS disallow. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_T3417_EXPIRED: EPS Mobility Management, T3417 expired. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_T3417_EXT_EXPIRED: EPS Mobility Management, T3417 ext expired. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_UL_DATA_CONFIRMATION_FAILURE_TXN: LTE Radio Resource Control, UL data confirmation failure, txn. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_UL_DATA_CONFIRMATION_FAILURE_HANDOVER: LTE Radio Resource Control, UL data confirmation failure, handover. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_UL_DATA_CONFIRMATION_FAILURE_CONN_REL: LTE Radio Resource Control, UL data confirmation failure, connection release. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_UL_DATA_CONFIRMATION_FAILURE_RLF: LTE Radio Resource Control, UL data confirmation failure, radio link failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_UL_DATA_CONFIRMATION_FAILURE_CTRL_NOT_CONN: LTE Radio Resource Control, UL data confirmation failure, ctrl not conn. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE: LTE Radio Resource Control, connection establishment failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_ABORTED: LTE Radio Resource Control, connection establishment failure, aborted. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_ACCESS_BARRED: LTE Radio Resource Control, connection establishment failure, access barred. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_CELL_RESELECTION: LTE Radio Resource Control, connection establishment failure, cell reselection. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_CONFIG_FAILURE: LTE Radio Resource Control, connection establishment failure, config failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_TIMER_EXPIRED: LTE Radio Resource Control, connection establishment failure, timer expired. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_LINK_FAILURE: LTE Radio Resource Control, connection establishment failure, link failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_NOT_CAMPED: LTE Radio Resource Control, connection establishment failure, not camped. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_SI_FAILURE: LTE Radio Resource Control, connection establishment failure, SI failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_REJECTED: LTE Radio Resource Control, connection establishment failure, rejected. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_NORMAL: LTE Radio Resource Control, connection release, normal. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_RLF: LTE Radio Resource Control, connection release, radio link failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_CRE_FAILURE: LTE Radio Resource Control, connection release, cre failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_OOS_DURING_CRE: LTE Radio Resource Control, connection release, oos during cre. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_ABORTED: LTE Radio Resource Control, connection release, aborted. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_SIB_READ_ERROR: LTE Radio Resource Control, connection release, SIB read error. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_DETACH_WITH_REATTACH_LTE_NW_DETACH: Detach with reattach LTE network detach. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_DETACH_WITHOUT_REATTACH_LTE_NW_DETACH: Detach without reattach LTE network detach. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_PROC_TIMEOUT: EPS Session Management, proc timeout. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_CONNECTION_ID: Invalid connection ID. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_NSAPI: Invalid Network Service Access Point Identifier (NSAPI). Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_PRI_NSAPI: Invalid primary Network Service Access Point Identifier (NSAPI). Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_FIELD: Invalid field. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RADIO_ACCESS_BEARER_SETUP_FAILURE: Radio Access Bearer (RAB) setup failure. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_ESTABLISH_MAX_TIMEOUT: PDP establishment max timeout. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_MODIFY_MAX_TIMEOUT: PDP modification max timeout. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_INACTIVE_MAX_TIMEOUT: PDP inactivity max timeout. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_LOWERLAYER_ERROR: PDP lower layer error. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PPD_UNKNOWN_REASON: PPD unknown reason. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_MODIFY_COLLISION: PDP modification collision. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_MBMS_REQUEST_COLLISION: PDP MBMS request collision. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MBMS_DUPLICATE: MBMS duplicate. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_SM_PS_DETACHED: SM PS detatched. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_SM_NO_RADIO_AVAILABLE: SM no radio available. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_SM_ABORT_SERVICE_NOT_AVAILABLE: SM abort service not available. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MESSAGE_EXCEEDS_MAX_L2_LIMIT: Message exceeds maximum Layer 2 limit. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_SM_NAS_SERVICE_REQUEST_FAILURE: SM NAS service request failure. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_ESTABLISHMENT_FAILURE_REQUEST_ERROR: RRC connection establishment failure; request error. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_ESTABLISHMENT_FAILURE_TAI_CHANGE: RRC connection establishment failure; TAI change. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_ESTABLISHMENT_FAILURE_RF_UNAVAILABLE: RRC connection establishment failure; RF unavailable. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_RELEASE_ABORTED_INTER_RAT_SUCCESS: RRC connection release; aborted Inter-RAT success. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_RELEASE_RLF_SEC_NOT_ACTIVE: RRC connection release; RLF security not active. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_RELEASE_INTER_RAT_TO_LTE_ABORTED: RRC connection release; Inter-RAT to LTE aborted. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_RELEASE_INTER_RAT_FROM_LTE_TO_GERAN_CCO_SUCCESS: RRC connection release; Inter-RAT from LTE to GERAN CCO success. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_RELEASE_INTER_RAT_FROM_LTE_TO_GERAN_CCO_ABORTED: RRC connection release; Inter-RAT from LTE to GERAN CCO aborted. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_IMSI_UNKNOWN_IN_HOME_SUBSCRIBER_SERVER: IMSI unknown in Home Subscriber Service (HSS). Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_IMEI_NOT_ACCEPTED: IMEI not accepted. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EPS_SERVICES_AND_NON_EPS_SERVICES_NOT_ALLOWED: EPS services and non-EPS services not allowed. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EPS_SERVICES_NOT_ALLOWED_IN_PLMN: EPS services not allowed in PLMN. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MSC_TEMPORARILY_NOT_REACHABLE: MSC temporarily not reachable. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CS_DOMAIN_NOT_AVAILABLE: Circuit-Switched domain not available. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_FAILURE: ESM failure. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MAC_FAILURE: MAC failure. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_SYNCHRONIZATION_FAILURE: Synchronization failure. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_UE_SECURITY_CAPABILITIES_MISMATCH: User Equipment (UE) security capabilities mismatch. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_SECURITY_MODE_REJECT_UNSPECIFIED: Security mode reject unspecified. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NON_EPS_AUTH_UNACCEPTABLE: Non-EPS authentication unaccpetable. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CS_FALLBACK_CALL_ESTABLISHMENT_NOT_ALLOWED: Circuit-Switched Fallback (CSFB) call establishment not allowed. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_EPS_BEARER_CONTEXT_ACTIVATED: No EPS bearer context activated. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_INVALID_STATE: EMM invalid state. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NAS_LAYER_FAILURE: NAS layer failure. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MULTI_PDN_NOT_ALLOWED: Multi-PDN not allowed. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMBMS_NOT_ENABLED: EMBMS not enabled. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PENDING_REDIAL_CALL_CLEANUP: Pending redial call cleanup. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMBMS_REGULAR_DEACTIVATION: EMBMS regular deactivation. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_TLB_REGULAR_DEACTIVATION: TLB regular deactivation. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LOWER_LAYER_REGISTRATION_FAILURE: Lower layer registration failure. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_DETACH_EPS_SERVICES_NOT_ALLOWED: Detach EPS services not allowed. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_SM_INTERNAL_PDP_DEACTIVATION: Session Management internal PDP deactivation. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONNECTION_DENY_GENERAL_OR_BUSY: CDMA/EVDO specific. Received Connection Deny. General or Network busy.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONNECTION_DENY_BILLING_OR_AUTHENTICATION_FAILURE: CDMA/EVDO specific. Received Connection Deny. Billing or Authentication failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_CHANGE: CDMA/EVDO specific. Change HDR.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_EXIT: CDMA/EVDO specific. Exit HDR.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_NO_SESSION: CDMA/EVDO specific. No HDR session.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_ORIGINATION_DURING_GPS_FIX: CDMA/EVDO specific. HDR call ended in favor of a GPS fix.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_CONNECTION_SETUP_TIMEOUT: CDMA/EVDO specific. Connection setup timeout.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_RELEASED_BY_CM: CDMA/EVDO specific. Released HDR call by call manager.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_COLLOCATED_ACQUISITION_FAILED: Failed to acquire co-located HDR for origination. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_OTASP_COMMIT_IN_PROGRESS: CDMA/EVDO specific. OTASP commit in progress. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_NO_HYBRID_SERVICE: CDMA/EVDO specific. No hybrid HDR service.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_NO_LOCK_GRANTED: CDMA/EVDO specific. No lock granted. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HOLD_OTHER_IN_PROGRESS: CDMA/EVDO specific. hold other in progress. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_FADE: CDMA/EVDO specific. Fade. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_ACCESS_FAILURE: CDMA/EVDO specific. Access failure. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_UNSUPPORTED_1X_PREV: Unsupported 1x PREV. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CLIENT_END: Client end.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_SERVICE: No service.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_FADE: Fade.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RELEASE_NORMAL: Release normal.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_ATTEMPT_IN_PROGRESS: Access attempt in progress.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_FAILURE: Access Failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_REDIRECTION_OR_HANDOFF: Redirection or handoff.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_OFFLINE: Offline. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMERGENCY_MODE: Emergency mode. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PHONE_IN_USE: Phone in use. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_MODE: Invalid mode. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_SIM_STATE: Invalid SIM state. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_COLLOCATED_HDR: No collocated HDR. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CALL_CONTROL_REJECTED: Call control rejected. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_DETACHED_PSM: EMM detached PSM. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_DUAL_SWITCH: Dual switch. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CALL_MANAGER: Call Manager. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_CLASS3_APN: Invalid class 3 APN. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MPLMN_IN_PROGRESS: MPLMN in progress. Since 1.36.
 *
 * Call manager specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_CM.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CDMA_LOCK                       = 500,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INTERCEPT                       = 501,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_REORDER                         = 502,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RELEASE_SO_REJECT               = 503,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMING_CALL                   = 504,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ALERT_STOP                      = 505,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACTIVATION                      = 506,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MAX_ACCESS_PROBES               = 507,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CCS_NOT_SUPPORTED_BY_BS         = 508,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_RESPONSE_FROM_BS             = 509,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_REJECTED_BY_BS                  = 510,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMPATIBLE                    = 511,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ALREADY_IN_TC                   = 512,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_USER_CALL_ORIGINATED_DURING_GPS = 513,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_USER_CALL_ORIGINATED_DURING_SMS = 514,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_CDMA_SERVICE                 = 515,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MC_ABORT                        = 516,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PSIST_NG                        = 517,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_UIM_NOT_PRESENT                 = 518,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RETRY_ORDER                     = 519,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_BLOCK                    = 520,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_BLOCK_ALL                = 521,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_IS707B_MAX_ACCESS_PROBES        = 522,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_THERMAL_EMERGENCY               = 523,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CALL_ORIGINATION_THROTTLED      = 524,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_USER_CALL_ORIGINATED            = 525,

    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONFERENCE_FAILED                      = 1000,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMING_REJECTED                      = 1001,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_GATEWAY_SERVICE                     = 1002,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_GPRS_CONTEXT                        = 1003,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ILLEGAL_MS                             = 1004,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ILLEGAL_ME                             = 1005,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_AND_NON_GPRS_SERVICES_NOT_ALLOWED = 1006,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_SERVICES_NOT_ALLOWED              = 1007,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MS_IDENTITY_NOT_DERIVED_BY_THE_NETWORK = 1008,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_IMPLICITLY_DETACHED                    = 1009,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PLMN_NOT_ALLOWED                       = 1010,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LA_NOT_ALLOWED                         = 1011,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_SERVICES_NOT_ALLOWED_IN_PLMN      = 1012,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_DUPLICATE                          = 1013,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_UE_RAT_CHANGE                          = 1014,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONGESTION                             = 1015,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_PDP_CONTEXT_ACTIVATED               = 1016,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_CLASS_DSAC_REJECTION            = 1017,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_ACTIVATE_MAX_RETRY_FAILED          = 1018,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RAB_FAILURE                            = 1019,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EPS_SERVICE_NOT_ALLOWED                = 1020,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_TRACKING_AREA_NOT_ALLOWED              = 1021,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ROAMING_NOT_ALLOWED_IN_TRACKING_AREA   = 1022,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_SUITABLE_CELLS_IN_TRACKING_AREA     = 1023,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NOT_AUTHORIZED_CLOSED_SUBSCRIBER_GROUP = 1024,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_UNKNOWN_EPS_BEARER_CONTEXT         = 1025,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_DRB_RELEASED_AT_RRC                    = 1026,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NAS_SIGNAL_CONNECTION_RELEASED         = 1027,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_DETACHED                           = 1028,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_ATTACH_FAILED                      = 1029,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_ATTACH_STARTED                     = 1030,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LTE_NAS_SERVICE_REQUEST_FAILED         = 1031,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_ACTIVE_DEDICATED_BEARER_REACTIVATED_BY_NW               = 1032,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_LOWER_LAYER_FAILURE                                     = 1033,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_SYNC_UP_WITH_NW                                         = 1034,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_NW_ACTIVATED_DEDICATED_BEARER_WITH_ID_OF_DEFAULT_BEARER = 1035,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_BAD_OTA_MESSAGE                                         = 1036,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_DS_REJECTED_CALL                                        = 1037,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_CONTEXT_TRANSFERRED_DUE_TO_IRAT                         = 1038,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_DS_EXPLICIT_DEACT                                           = 1039,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_LOCAL_CAUSE_NONE                                        = 1040,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LTE_NAS_SERVICE_REQUEST_FAILED_NO_THROTTLE                  = 1041,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACL_FAILURE                                                 = 1042,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LTE_NAS_SERVICE_REQUEST_FAILED_DS_DISALLOW                  = 1043,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_T3417_EXPIRED                                           = 1044,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_T3417_EXT_EXPIRED                                       = 1045,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_UL_DATA_CONFIRMATION_FAILURE_TXN                       = 1046,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_UL_DATA_CONFIRMATION_FAILURE_HANDOVER                  = 1047,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_UL_DATA_CONFIRMATION_FAILURE_CONN_REL                  = 1048,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_UL_DATA_CONFIRMATION_FAILURE_RLF                       = 1049,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_UL_DATA_CONFIRMATION_FAILURE_CTRL_NOT_CONN             = 1050,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE                       = 1051,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_ABORTED               = 1052,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_ACCESS_BARRED         = 1053,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_CELL_RESELECTION      = 1054,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_CONFIG_FAILURE        = 1055,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_TIMER_EXPIRED         = 1056,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_LINK_FAILURE          = 1057,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_NOT_CAMPED            = 1058,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_SI_FAILURE            = 1059,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_ESTABLISHMENT_FAILURE_REJECTED              = 1060,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_NORMAL         = 1061,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_RLF            = 1062,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_CRE_FAILURE    = 1063,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_OOS_DURING_CRE = 1064,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_ABORTED        = 1065,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LRRC_CONNECTION_RELEASE_SIB_READ_ERROR = 1066,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_DETACH_WITH_REATTACH_LTE_NW_DETACH     = 1067,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_DETACH_WITHOUT_REATTACH_LTE_NW_DETACH  = 1068,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_PROC_TIMEOUT                       = 1069,

    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_CONNECTION_ID                  = 1070,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_NSAPI                          = 1071,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_PRI_NSAPI                      = 1072,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_FIELD                          = 1073,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RADIO_ACCESS_BEARER_SETUP_FAILURE      = 1074,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_ESTABLISH_MAX_TIMEOUT              = 1075,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_MODIFY_MAX_TIMEOUT                 = 1076,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_INACTIVE_MAX_TIMEOUT               = 1077,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_LOWERLAYER_ERROR                   = 1078,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PPD_UNKNOWN_REASON                     = 1079,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_MODIFY_COLLISION                   = 1080,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_MBMS_REQUEST_COLLISION             = 1081,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MBMS_DUPLICATE                         = 1082,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_SM_PS_DETACHED                         = 1083,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_SM_NO_RADIO_AVAILABLE                  = 1084,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_SM_ABORT_SERVICE_NOT_AVAILABLE         = 1085,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MESSAGE_EXCEEDS_MAX_L2_LIMIT           = 1086,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_SM_NAS_SERVICE_REQUEST_FAILURE         = 1087,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_ESTABLISHMENT_FAILURE_REQUEST_ERROR             = 1088,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_ESTABLISHMENT_FAILURE_TAI_CHANGE                = 1089,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_ESTABLISHMENT_FAILURE_RF_UNAVAILABLE            = 1090,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_RELEASE_ABORTED_INTER_RAT_SUCCESS               = 1091,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_RELEASE_RLF_SEC_NOT_ACTIVE                      = 1092,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_RELEASE_INTER_RAT_TO_LTE_ABORTED                = 1093,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_RELEASE_INTER_RAT_FROM_LTE_TO_GERAN_CCO_SUCCESS = 1094,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RRC_CONNECTION_RELEASE_INTER_RAT_FROM_LTE_TO_GERAN_CCO_ABORTED = 1095,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_IMSI_UNKNOWN_IN_HOME_SUBSCRIBER_SERVER        = 1096,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_IMEI_NOT_ACCEPTED                             = 1097,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EPS_SERVICES_AND_NON_EPS_SERVICES_NOT_ALLOWED = 1098,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EPS_SERVICES_NOT_ALLOWED_IN_PLMN              = 1099,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MSC_TEMPORARILY_NOT_REACHABLE                 = 1100,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CS_DOMAIN_NOT_AVAILABLE                       = 1101,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ESM_FAILURE                                   = 1102,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MAC_FAILURE                                   = 1103,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_SYNCHRONIZATION_FAILURE                       = 1104,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_UE_SECURITY_CAPABILITIES_MISMATCH             = 1105,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_SECURITY_MODE_REJECT_UNSPECIFIED              = 1106,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NON_EPS_AUTH_UNACCEPTABLE                     = 1107,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CS_FALLBACK_CALL_ESTABLISHMENT_NOT_ALLOWED    = 1108,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_EPS_BEARER_CONTEXT_ACTIVATED               = 1109,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_INVALID_STATE                             = 1110,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NAS_LAYER_FAILURE                             = 1111,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MULTI_PDN_NOT_ALLOWED                         = 1112,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMBMS_NOT_ENABLED                             = 1113,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PENDING_REDIAL_CALL_CLEANUP                   = 1114,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMBMS_REGULAR_DEACTIVATION                    = 1115,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_TLB_REGULAR_DEACTIVATION                      = 1116,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LOWER_LAYER_REGISTRATION_FAILURE              = 1117,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_DETACH_EPS_SERVICES_NOT_ALLOWED               = 1118,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_SM_INTERNAL_PDP_DEACTIVATION                  = 1119,

    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONNECTION_DENY_GENERAL_OR_BUSY                   = 1500,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONNECTION_DENY_BILLING_OR_AUTHENTICATION_FAILURE = 1501,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_CHANGE                                        = 1502,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_EXIT                                          = 1503,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_NO_SESSION                                    = 1504,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_ORIGINATION_DURING_GPS_FIX                    = 1505,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_CONNECTION_SETUP_TIMEOUT                      = 1506,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_RELEASED_BY_CM                                = 1507,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_COLLOCATED_ACQUISITION_FAILED                 = 1508,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_OTASP_COMMIT_IN_PROGRESS                          = 1509,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_NO_HYBRID_SERVICE                             = 1510,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_NO_LOCK_GRANTED                               = 1511,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HOLD_OTHER_IN_PROGRESS                            = 1512,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_FADE                                          = 1513,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_ACCESS_FAILURE                                = 1514,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_UNSUPPORTED_1X_PREV                               = 1515,

    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CLIENT_END                 = 2000,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_SERVICE                 = 2001,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_FADE                       = 2002,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RELEASE_NORMAL             = 2003,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_ATTEMPT_IN_PROGRESS = 2004,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_FAILURE             = 2005,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_REDIRECTION_OR_HANDOFF     = 2006,

    QMI_WDS_VERBOSE_CALL_END_REASON_CM_OFFLINE               = 2500,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMERGENCY_MODE        = 2501,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PHONE_IN_USE          = 2502,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_MODE          = 2503,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_SIM_STATE     = 2504,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_COLLOCATED_HDR     = 2505,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CALL_CONTROL_REJECTED = 2506,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_EMM_DETACHED_PSM      = 2507,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_DUAL_SWITCH           = 2508,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CALL_MANAGER          = 2509,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INVALID_CLASS3_APN    = 2510,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MPLMN_IN_PROGRESS     = 2511,
} QmiWdsVerboseCallEndReasonCm;

/**
 * QmiWdsVerboseCallEndReason3gpp:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPERATOR_DETERMINED_BARRING: Operator-determined barring.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_LLC_SNDCP_FAILURE: LLC or SNDCP failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INSUFFICIENT_RESOURCES: Insufficient resources.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_APN:  Unknown or missing APN.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_PDP: Unknown PDP address or type.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_AUTHENTICATION_FAILED: Authentication failed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_GGSN_REJECT: Activation rejected by GGSN.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_ACTIVATION_REJECT: Activation rejected.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_NOT_SUPPORTED:  Service option not supported.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_UNSUBSCRIBED: Service option not subscribed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_TEMPORARILY_OUT_OF_ORDER: Service option temporarily out of order.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_NSAPI_ALREADY_USED: NSAPI already used.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_REGULAR_DEACTIVATION: Regular PDP context deactivation.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_QOS_NOT_ACCEPTED: QoS not accepted.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_NETWORK_FAILURE: Network failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_REATTACH_REQUIRED: Reattach required.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FEATURE_NOT_SUPPORTED: Feature not supported.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_TFT_SEMANTIC_ERROR: Semantic error in TFT operation.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_TFT_SYNTAX_ERROR: Syntax error in TFT operation.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_PDP_CONTEXT: Unknown PDP context.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FILTER_SEMANTIC_ERROR: Semantic error in packet filters.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FILTER_SYNTAX_ERROR: Syntax error in packet filters.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PDP_WITHOUT_ACTIVE_TFT: PDP context without TFT activated.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_IPV4_ONLY_ALLOWED: IPv4 only allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_IPV6_ONLY_ALLOWED: IPv6 only allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_SINGLE_ADDRESS_BEARER_ONLY: Single address bearer only.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_ESM_INFO_NOT_RECEIVED: ESM information not received.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PDN_CONNECTION_DOES_NOT_EXIST: PDN connection does not exist.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MULTIPLE_CONNECTION_TO_SAME_PDN_NOT_ALLOWED: Multiple connection to same PDN not allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_TRANSACTION_ID: Invalid transaction ID.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_INCORRECT_SEMANTIC: Message incorrect semantically.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_MANDATORY_INFO:  Invalid mandatory information.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_TYPE_UNSUPPORTED: Message type not implemented.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_TYPE_NONCOMPATIBLE_STATE: Message not compatible with state.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_INFO_ELEMENT: Information element unknown.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_CONDITIONAL_IE_ERROR: Conditional IE error.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_AND_PROTOCOL_STATE_UNCOMPATIBLE: Message and protocol state uncompatible.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PROTOCOL_ERROR: Protocol error.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_APN_TYPE_CONFLICT: APN type conflict.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_PROXY_CALL_SESSION_CONTROL_FUNCTION_ADDRESS: Invalid proxy call session control function address. Since 1.24.10.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INTERNAL_CALL_PREEMPTED_BY_HIGH_PRIORITY_APN: Internal call pre-empted by high-priority APN. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_EMM_ACCESS_BARRED: EMM access barred. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_EMERGENCY_INTERFACE_ONLY: Emergency interface only. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INTERFACE_MISMATCH: Interface mismatch. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_COMPANION_INTERFACE_IN_USE: Companion interface in use. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_IP_ADDRESS_MISMATCH: IP address mismatch. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INTERFACE_AND_POLICY_FAMILY_MISMATCH: Interface and policy family mismatch. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_EMM_ACCESS_BARRED_INFINITE_RETRY: EMM access barred; infinit retry. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_AUTHENTICATION_FAILURE_ON_EMERGENCY_CALL: Authentication failure on emergency call. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_DNS_ADDRESS: Invalid DNS address. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_PROXY_CALL_SESSION_CONTROL_FUNCTION_DNS_ADDRESS: Invalid Proxy Call Session Control Function (PCSCF) DNS address. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_TEST_LOOPBACK_MODE_A_OR_B_ENABLED: Test loopback mode A or B enabled. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_EMM_ACCESS_BARRED_EAB: EMM access barred EAB. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_CALL_PREEMPTED_BY_EMERGENCY_APN: Call pre-empted by emergency APN. Since 1.36.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UE_INIT_DETACH_OR_DISCONNECT: User Equipment (UE) initialization detach or disconnect. Since 1.36.
 *
 * 3GPP-specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_3GPP.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_verbose_call_end_reason_3gpp,since=1.0 >*/
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPERATOR_DETERMINED_BARRING = 8,

    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_LLC_SNDCP_FAILURE               = 25,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INSUFFICIENT_RESOURCES          = 26,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_APN                     = 27,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_PDP                     = 28,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_AUTHENTICATION_FAILED           = 29,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_GGSN_REJECT                     = 30,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_ACTIVATION_REJECT               = 31,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_NOT_SUPPORTED            = 32,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_UNSUBSCRIBED             = 33,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_TEMPORARILY_OUT_OF_ORDER = 34,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_NSAPI_ALREADY_USED              = 35,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_REGULAR_DEACTIVATION            = 36,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_QOS_NOT_ACCEPTED                = 37,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_NETWORK_FAILURE                 = 38,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_REATTACH_REQUIRED               = 39,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FEATURE_NOT_SUPPORTED           = 40,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_TFT_SEMANTIC_ERROR              = 41,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_TFT_SYNTAX_ERROR                = 42,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_PDP_CONTEXT             = 43,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FILTER_SEMANTIC_ERROR           = 44,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FILTER_SYNTAX_ERROR             = 45,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PDP_WITHOUT_ACTIVE_TFT          = 46,

    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_IPV4_ONLY_ALLOWED                           = 50,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_IPV6_ONLY_ALLOWED                           = 51,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_SINGLE_ADDRESS_BEARER_ONLY                  = 52,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_ESM_INFO_NOT_RECEIVED                       = 53,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PDN_CONNECTION_DOES_NOT_EXIST               = 54,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MULTIPLE_CONNECTION_TO_SAME_PDN_NOT_ALLOWED = 55,

    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_TRANSACTION_ID = 81,

    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_INCORRECT_SEMANTIC              = 95,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_MANDATORY_INFO                  = 96,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_TYPE_UNSUPPORTED                = 97,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_TYPE_NONCOMPATIBLE_STATE        = 98,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_INFO_ELEMENT                    = 99,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_CONDITIONAL_IE_ERROR                    = 100,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_AND_PROTOCOL_STATE_UNCOMPATIBLE = 101,

    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PROTOCOL_ERROR                                       = 111,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_APN_TYPE_CONFLICT                                    = 112,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_PROXY_CALL_SESSION_CONTROL_FUNCTION_ADDRESS  = 113,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INTERNAL_CALL_PREEMPTED_BY_HIGH_PRIORITY_APN         = 114,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_EMM_ACCESS_BARRED                                    = 115,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_EMERGENCY_INTERFACE_ONLY                             = 116,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INTERFACE_MISMATCH                                   = 117,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_COMPANION_INTERFACE_IN_USE                           = 118,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_IP_ADDRESS_MISMATCH                                  = 119,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INTERFACE_AND_POLICY_FAMILY_MISMATCH                 = 120,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_EMM_ACCESS_BARRED_INFINITE_RETRY                     = 121,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_AUTHENTICATION_FAILURE_ON_EMERGENCY_CALL             = 122,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_DNS_ADDRESS                                  = 123,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_PROXY_CALL_SESSION_CONTROL_FUNCTION_DNS_ADDRESS = 124,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_TEST_LOOPBACK_MODE_A_OR_B_ENABLED                       = 125,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_EMM_ACCESS_BARRED_EAB                                   = 126,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_CALL_PREEMPTED_BY_EMERGENCY_APN                         = 127,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UE_INIT_DETACH_OR_DISCONNECT                            = 128,
} QmiWdsVerboseCallEndReason3gpp;

/**
 * QmiWdsVerboseCallEndReasonPpp:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_UNKNOWN: Unknown error.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_TIMEOUT: Timeout.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_AUTHENTICATION_FAILURE: Authentication failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_OPTION_MISMATCH: Option mismatch.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_PAP_FAILURE: PAP failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_CHAP_FAILURE: CHAP failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_CLOSE_IN_PROGRESS: Close in progress. Since 1.24.10.
 *
 * PPP-specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_PPP.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_UNKNOWN = -1,

    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_TIMEOUT                = 1,
    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_AUTHENTICATION_FAILURE = 2,
    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_OPTION_MISMATCH        = 3,

    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_PAP_FAILURE       = 31,
    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_CHAP_FAILURE      = 32,
    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_CLOSE_IN_PROGRESS = 33,

} QmiWdsVerboseCallEndReasonPpp;

/**
 * QmiWdsVerboseCallEndReasonEhrpd:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_SUBSCRIPTION_LIMITED_TO_IPV4: Subscription limited to IPv4.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_SUBSCRIPTION_LIMITED_TO_IPV6: Subscription limited to IPv6.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_TIMEOUT: VSNCP specific. timeout.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_FAILURE: VSNCP specific. failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_GENERAL_ERROR: VSNCP specific. 3GPP2 general error.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_UNAUTHENTICATED_APN: VSNCP specific. 3GPP2 unauthenticated APN.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_LIMIT_EXCEEDED: VSNCP specific. 3GPP2 PDN limit exceeded.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_NO_PDN_GATEWAY: VSNCP specific. 3GPP2 no PDN gateway.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_GATEWAY_UNREACHABLE: VSNCP specific. 3GPP2 PDN gateway unreachable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_GATEWAY_REJECTED: VSNCP specific. 3GPP2 PDN gateway rejected.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_INSUFFICIENT_PARAMETERS: VSNCP specific. 3GPP2 insufficient parameters.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_RESOURCE_UNAVAILABLE: VSNCP specific. 3GPP2 resource unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_ADMINISTRATIVELY_PROHIBITED: VSNCP specific. 3GPP2 administratively prohibited.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_ID_IN_USE: VSNCP specific. 3GPP2 PDN ID in use.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_SUBSCRIPTION_LIMITATION: VSNCP specific. 3GPP2 subscription limitation.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_EXISTS_FOR_THIS_APN: VSNCP specific. 3GPP2 PDN exists for this APN.
 *
 * eHRPD-specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_EHRPD.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_SUBSCRIPTION_LIMITED_TO_IPV4 = 1,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_SUBSCRIPTION_LIMITED_TO_IPV6 = 2,

    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_TIMEOUT                           = 4,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_FAILURE                           = 5,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_GENERAL_ERROR               = 6,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_UNAUTHENTICATED_APN         = 7,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_LIMIT_EXCEEDED          = 8,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_NO_PDN_GATEWAY              = 9,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_GATEWAY_UNREACHABLE     = 10,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_GATEWAY_REJECTED        = 11,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_INSUFFICIENT_PARAMETERS     = 12,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_RESOURCE_UNAVAILABLE        = 13,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_ADMINISTRATIVELY_PROHIBITED = 14,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_ID_IN_USE               = 15,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_SUBSCRIPTION_LIMITATION     = 16,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_EXISTS_FOR_THIS_APN     = 17
} QmiWdsVerboseCallEndReasonEhrpd;

/**
 * QmiWdsVerboseCallEndReasonIpv6:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_IPV6_PREFIX_UNAVAILABLE: Prefix unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_IPV6_HRPD_IPV6_DISABLED: HRDP IPv6 disabled.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_IPV6_DISABLED: IPv6 disabled. Since 1.24.10.
 *
 * IPv6-specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_IPV6.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_VERBOSE_CALL_END_REASON_IPV6_PREFIX_UNAVAILABLE = 1,
    QMI_WDS_VERBOSE_CALL_END_REASON_IPV6_HRPD_IPV6_DISABLED = 2,
    QMI_WDS_VERBOSE_CALL_END_REASON_IPV6_DISABLED           = 3,
} QmiWdsVerboseCallEndReasonIpv6;

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Packet Service Status' message */

/**
 * QmiWdsConnectionStatus:
 * @QMI_WDS_CONNECTION_STATUS_UNKNOWN: Unknown status.
 * @QMI_WDS_CONNECTION_STATUS_DISCONNECTED: Network is disconnected
 * @QMI_WDS_CONNECTION_STATUS_CONNECTED: Network is connected.
 * @QMI_WDS_CONNECTION_STATUS_SUSPENDED: Network connection is suspended.
 * @QMI_WDS_CONNECTION_STATUS_AUTHENTICATING: Network authentication is ongoing.
 *
 * Status of the network connection.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_CONNECTION_STATUS_UNKNOWN        = 0,
    QMI_WDS_CONNECTION_STATUS_DISCONNECTED   = 1,
    QMI_WDS_CONNECTION_STATUS_CONNECTED      = 2,
    QMI_WDS_CONNECTION_STATUS_SUSPENDED      = 3,
    QMI_WDS_CONNECTION_STATUS_AUTHENTICATING = 4
} QmiWdsConnectionStatus;

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Data Bearer Technology' message */

/**
 * QmiWdsDataBearerTechnology:
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_UNKNOWN: Unknown.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_CDMA20001X: CDMA2000 1x.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_1xEVDO: CDMA2000 HRPD 1xEVDO.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_GSM: GSM.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_UMTS: UMTS.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_1xEVDO_REVA: CDMA2000 HRPD 1xEVDO RevA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_EDGE: EDGE.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPA: HSDPA and WCDMA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSUPA: WCDMA and HSUPA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPA_HSUPDA: HSDPA and HSUPA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_LTE: LTE.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_EHRPD: CDMA2000 eHRPD.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS: HSDPA+ and WCDMA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_HSUPA: HSDPA+ and HSUPA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_DCHSDPAPLUS: DC-HSDPA+ and WCDMA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_DCHSDPAPLUS_HSUPA: DC-HSDPA+ and HSUPA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_PLUS_64QAM: HSDPA+ with 64QAM. Since 1.18.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_PLUS_64QAM_HSUPA: HSDPA+ with 64QAM and HSUPA. Since 1.18.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA: TD-SCDMA. Since 1.18.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA_HSDPA: TD-SCDMA and HSDPA. Since 1.18.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA_HSUPA: TD-SCDMA and HSUPA. Since 1.18.
 *
 * Data bearer technology.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_DATA_BEARER_TECHNOLOGY_UNKNOWN                    = -1,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_CDMA20001X                 = 0x01,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_1xEVDO                     = 0x02,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_GSM                        = 0x03,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_UMTS                       = 0x04,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_1xEVDO_REVA                = 0x05,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_EDGE                       = 0x06,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPA                      = 0x07,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSUPA                      = 0x08,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPA_HSUPDA               = 0x09,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_LTE                        = 0x0A,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_EHRPD                      = 0x0B,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS                  = 0x0C,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_HSUPA            = 0x0D,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_DCHSDPAPLUS                = 0x0E,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_DCHSDPAPLUS_HSUPA          = 0x0F,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_PLUS_64QAM       = 0x10,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_PLUS_64QAM_HSUPA = 0x11,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA                    = 0x12,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA_HSDPA              = 0x13,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA_HSUPA              = 0x14,
} QmiWdsDataBearerTechnology;

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Current Data Bearer Technology' message */

/**
 * QmiWdsNetworkType:
 * @QMI_WDS_NETWORK_TYPE_UNKNOWN: Unknown.
 * @QMI_WDS_NETWORK_TYPE_3GPP2: 3GPP2 network type.
 * @QMI_WDS_NETWORK_TYPE_3GPP: 3GPP network type.
 *
 * Network type of the data bearer.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_NETWORK_TYPE_UNKNOWN = 0,
    QMI_WDS_NETWORK_TYPE_3GPP2   = 1,
    QMI_WDS_NETWORK_TYPE_3GPP    = 2
} QmiWdsNetworkType;

/**
 * QmiWdsDataSystemNetworkType:
 * @QMI_WDS_DATA_SYSTEM_NETWORK_TYPE_3GPP2: 3GPP2 network type.
 * @QMI_WDS_DATA_SYSTEM_NETWORK_TYPE_3GPP: 3GPP network type.
 *
 * Network type of the data system.
 *
 * Since: 1.14
 */
typedef enum { /*< since=1.14 >*/
    QMI_WDS_DATA_SYSTEM_NETWORK_TYPE_3GPP  = 0,
    QMI_WDS_DATA_SYSTEM_NETWORK_TYPE_3GPP2 = 1,
} QmiWdsDataSystemNetworkType;

/**
 * QmiWdsDataSystem:
 * @QMI_WDS_DATA_SYSTEM_UNKNOWN: unknown
 * @QMI_WDS_DATA_SYSTEM_CDMA1X: CDMA 1x
 * @QMI_WDS_DATA_SYSTEM_CDMA1X_EVDO: CDMA 1x EVDO
 * @QMI_WDS_DATA_SYSTEM_GPRS: GPRS
 * @QMI_WDS_DATA_SYSTEM_WCDMA: WCDMA
 * @QMI_WDS_DATA_SYSTEM_LTE: LTE
 * @QMI_WDS_DATA_SYSTEM_TDSCDMA: TD-SCDMA.
 *
 * Data system type.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_DATA_SYSTEM_UNKNOWN     = 0,
    QMI_WDS_DATA_SYSTEM_CDMA1X      = 1,
    QMI_WDS_DATA_SYSTEM_CDMA1X_EVDO = 2,
    QMI_WDS_DATA_SYSTEM_GPRS        = 3,
    QMI_WDS_DATA_SYSTEM_WCDMA       = 4,
    QMI_WDS_DATA_SYSTEM_LTE         = 5,
    QMI_WDS_DATA_SYSTEM_TDSCDMA     = 6,
} QmiWdsDataSystem;

/**
 * QmiWdsRat3gpp2:
 * @QMI_WDS_RAT_3GPP2_NONE: Unknown, to be ignored.
 * @QMI_WDS_RAT_3GPP2_CDMA1X: CDMA 1x.
 * @QMI_WDS_RAT_3GPP2_EVDO_REV0: EVDO Rev0.
 * @QMI_WDS_RAT_3GPP2_EVDO_REVA: EVDO RevA.
 * @QMI_WDS_RAT_3GPP2_EVDO_REVB: EVDO RevB.
 * @QMI_WDS_RAT_3GPP2_EHRPD: eHRPD. Since 1.18.
 * @QMI_WDS_RAT_3GPP2_FMC: FMC. Since 1.18.
 * @QMI_WDS_RAT_3GPP2_NULL_BEARER: No bearer.
 *
 * Flags specifying the 3GPP2-specific Radio Access Technology, when the data
 * bearer network type is @QMI_WDS_NETWORK_TYPE_3GPP2.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_rat_3gpp2,since=1.0 >*/
    QMI_WDS_RAT_3GPP2_NONE        = 0,
    QMI_WDS_RAT_3GPP2_CDMA1X      = 1 << 0,
    QMI_WDS_RAT_3GPP2_EVDO_REV0   = 1 << 1,
    QMI_WDS_RAT_3GPP2_EVDO_REVA   = 1 << 2,
    QMI_WDS_RAT_3GPP2_EVDO_REVB   = 1 << 3,
    QMI_WDS_RAT_3GPP2_EHRPD       = 1 << 4,
    QMI_WDS_RAT_3GPP2_FMC         = 1 << 5,
    QMI_WDS_RAT_3GPP2_NULL_BEARER = 1 << 15
} QmiWdsRat3gpp2;

/**
 * QmiWdsRat3gpp:
 * @QMI_WDS_RAT_3GPP_NONE: Unknown, to be ignored.
 * @QMI_WDS_RAT_3GPP_WCDMA: WCDMA.
 * @QMI_WDS_RAT_3GPP_GPRS: GPRS.
 * @QMI_WDS_RAT_3GPP_HSDPA: HSDPA.
 * @QMI_WDS_RAT_3GPP_HSUPA: HSUPA.
 * @QMI_WDS_RAT_3GPP_EDGE: EDGE.
 * @QMI_WDS_RAT_3GPP_LTE: LTE.
 * @QMI_WDS_RAT_3GPP_HSDPAPLUS: HSDPA+.
 * @QMI_WDS_RAT_3GPP_DCHSDPAPLUS: DC-HSDPA+.
 * @QMI_WDS_RAT_3GPP_64QAM: 64QAM. Since 1.18.
 * @QMI_WDS_RAT_3GPP_TDSCDMA: TD-SCDMA. Since 1.18.
 * @QMI_WDS_RAT_3GPP_5GNR: 5GNR. Since 1.36.
 * @QMI_WDS_RAT_3GPP_NULL_BEARER: No bearer.
 *
 * Flags specifying the 3GPP-specific Radio Access Technology, when the data
 * bearer network type is @QMI_WDS_NETWORK_TYPE_3GPP.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_rat_3gpp,since=1.0 >*/
    QMI_WDS_RAT_3GPP_NONE        = 0,
    QMI_WDS_RAT_3GPP_WCDMA       = 1 << 0,
    QMI_WDS_RAT_3GPP_GPRS        = 1 << 1,
    QMI_WDS_RAT_3GPP_HSDPA       = 1 << 2,
    QMI_WDS_RAT_3GPP_HSUPA       = 1 << 3,
    QMI_WDS_RAT_3GPP_EDGE        = 1 << 4,
    QMI_WDS_RAT_3GPP_LTE         = 1 << 5,
    QMI_WDS_RAT_3GPP_HSDPAPLUS   = 1 << 6,
    QMI_WDS_RAT_3GPP_DCHSDPAPLUS = 1 << 7,
    QMI_WDS_RAT_3GPP_64QAM       = 1 << 8,
    QMI_WDS_RAT_3GPP_TDSCDMA     = 1 << 9,
    QMI_WDS_RAT_3GPP_5GNR        = 1 << 10,
    QMI_WDS_RAT_3GPP_NULL_BEARER = 1 << 15
} QmiWdsRat3gpp;

/**
 * QmiWdsSoCdma1x:
 * @QMI_WDS_SO_CDMA1X_NONE: Unknown, to be ignored.
 * @QMI_WDS_SO_CDMA1X_IS95: IS95.
 * @QMI_WDS_SO_CDMA1X_IS2000: IS2000.
 * @QMI_WDS_SO_CDMA1X_IS2000_REL_A: IS2000 RelA.
 *
 * Flags specifying the Service Option when the bearer network type is
 * @QMI_WDS_NETWORK_TYPE_3GPP2 and when the Radio Access Technology mask
 * contains @QMI_WDS_RAT_3GPP2_CDMA1X.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_SO_CDMA1X_NONE         = 0,
    QMI_WDS_SO_CDMA1X_IS95         = 1 << 0,
    QMI_WDS_SO_CDMA1X_IS2000       = 1 << 1,
    QMI_WDS_SO_CDMA1X_IS2000_REL_A = 1 << 2
} QmiWdsSoCdma1x;

/**
 * QmiWdsSoEvdoRev0:
 * @QMI_WDS_SO_EVDO_REV0_NONE: Unknown, to be ignored.
 * @QMI_WDS_SO_EVDO_REV0_DPA: DPA.
 *
 * Flags specifying the Service Option when the bearer network type is
 * @QMI_WDS_NETWORK_TYPE_3GPP2 and when the Radio Access Technology mask
 * contains @QMI_WDS_RAT_3GPP2_EVDO_REV0.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_SO_EVDO_REV0_NONE = 0,
    QMI_WDS_SO_EVDO_REV0_DPA  = 1 << 0,
} QmiWdsSoEvdoRev0;

/**
 * QmiWdsSoEvdoRevA:
 * @QMI_WDS_SO_EVDO_REVA_NONE: Unknown, to be ignored.
 * @QMI_WDS_SO_EVDO_REVA_DPA: DPA.
 * @QMI_WDS_SO_EVDO_REVA_MFPA: MFPA.
 * @QMI_WDS_SO_EVDO_REVA_EMPA: EMPA.
 * @QMI_WDS_SO_EVDO_REVA_EMPA_EHRPD: EMPA eHRPD.
 *
 * Flags specifying the Service Option when the bearer network type is
 * @QMI_WDS_NETWORK_TYPE_3GPP2 and when the Radio Access Technology mask
 * contains @QMI_WDS_RAT_3GPP2_EVDO_REVA.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_so_evdo_reva,since=1.0 >*/
    QMI_WDS_SO_EVDO_REVA_NONE       = 0,
    QMI_WDS_SO_EVDO_REVA_DPA        = 1 << 0,
    QMI_WDS_SO_EVDO_REVA_MFPA       = 1 << 1,
    QMI_WDS_SO_EVDO_REVA_EMPA       = 1 << 2,
    QMI_WDS_SO_EVDO_REVA_EMPA_EHRPD = 1 << 3,
} QmiWdsSoEvdoRevA;

/**
 * QmiWdsSoEvdoRevB:
 * @QMI_WDS_SO_EVDO_REVB_NONE: Unknown, to be ignored.
 * @QMI_WDS_SO_EVDO_REVB_DPA: DPA.
 * @QMI_WDS_SO_EVDO_REVB_MFPA: MFPA.
 * @QMI_WDS_SO_EVDO_REVB_EMPA: EMPA.
 * @QMI_WDS_SO_EVDO_REVB_EMPA_EHRPD: EMPA eHRPD.
 * @QMI_WDS_SO_EVDO_REVB_MMPA: MMPA.
 * @QMI_WDS_SO_EVDO_REVB_MMPA_EHRPD: MMPA eHRPD.
 *
 * Flags specifying the Service Option when the bearer network type is
 * @QMI_WDS_NETWORK_TYPE_3GPP2 and when the Radio Access Technology mask
 * contains @QMI_WDS_RAT_3GPP2_EVDO_REVB.
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_so_evdo_revb,since=1.18 >*/
    QMI_WDS_SO_EVDO_REVB_NONE       = 0,
    QMI_WDS_SO_EVDO_REVB_DPA        = 1 << 0,
    QMI_WDS_SO_EVDO_REVB_MFPA       = 1 << 1,
    QMI_WDS_SO_EVDO_REVB_EMPA       = 1 << 2,
    QMI_WDS_SO_EVDO_REVB_EMPA_EHRPD = 1 << 3,
    QMI_WDS_SO_EVDO_REVB_MMPA       = 1 << 4,
    QMI_WDS_SO_EVDO_REVB_MMPA_EHRPD = 1 << 5,
} QmiWdsSoEvdoRevB;

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Current Settings' message */

/**
 * QmiWdsRequestedSettings:
 * @QMI_WDS_REQUESTED_SETTINGS_NONE: no settings.
 * @QMI_WDS_REQUESTED_SETTINGS_PROFILE_ID: profile ID.
 * @QMI_WDS_REQUESTED_SETTINGS_PROFILE_NAME: profile name.
 * @QMI_WDS_REQUESTED_SETTINGS_PDP_TYPE: PDP context type.
 * @QMI_WDS_REQUESTED_SETTINGS_APN_NAME: APN name.
 * @QMI_WDS_REQUESTED_SETTINGS_DNS_ADDRESS: DNS server addresses.
 * @QMI_WDS_REQUESTED_SETTINGS_GRANTED_QOS: granted QoS.
 * @QMI_WDS_REQUESTED_SETTINGS_USERNAME: username.
 * @QMI_WDS_REQUESTED_SETTINGS_AUTH_PROTOCOL: authentication protocol, ie PAP/CHAP/none.
 * @QMI_WDS_REQUESTED_SETTINGS_IP_ADDRESS: IP address.
 * @QMI_WDS_REQUESTED_SETTINGS_GATEWAY_INFO: gateway address.
 * @QMI_WDS_REQUESTED_SETTINGS_PCSCF_ADDRESS: PCSCF address.
 * @QMI_WDS_REQUESTED_SETTINGS_PCSCF_SERVER_ADDRESS_LIST: PCSCF server address list.
 * @QMI_WDS_REQUESTED_SETTINGS_PCSCF_DOMAIN_NAME_LIST: PCSCF domain name list.
 * @QMI_WDS_REQUESTED_SETTINGS_MTU: MTU.
 * @QMI_WDS_REQUESTED_SETTINGS_DOMAIN_NAME_LIST: domain name list.
 * @QMI_WDS_REQUESTED_SETTINGS_IP_FAMILY: IP family, ie IPv4 or IPv6.
 * @QMI_WDS_REQUESTED_SETTINGS_IMCN_FLAG: IMCN flag.
 * @QMI_WDS_REQUESTED_SETTINGS_EXTENDED_TECHNOLOGY: extended technology info.
 * @QMI_WDS_REQUESTED_SETTINGS_OPERATOR_RESERVED_PCO: operator reserved protocol configuration option.
 *
 * Flags specifying requested WDS settings.
 *
 * Since: 1.32
 */
typedef enum { /*< since=1.32 >*/
    QMI_WDS_REQUESTED_SETTINGS_NONE                      = 0,
    QMI_WDS_REQUESTED_SETTINGS_PROFILE_ID                = 1 << 0,
    QMI_WDS_REQUESTED_SETTINGS_PROFILE_NAME              = 1 << 1,
    QMI_WDS_REQUESTED_SETTINGS_PDP_TYPE                  = 1 << 2,
    QMI_WDS_REQUESTED_SETTINGS_APN_NAME                  = 1 << 3,
    QMI_WDS_REQUESTED_SETTINGS_DNS_ADDRESS               = 1 << 4,
    QMI_WDS_REQUESTED_SETTINGS_GRANTED_QOS               = 1 << 5,
    QMI_WDS_REQUESTED_SETTINGS_USERNAME                  = 1 << 6,
    QMI_WDS_REQUESTED_SETTINGS_AUTH_PROTOCOL             = 1 << 7,
    QMI_WDS_REQUESTED_SETTINGS_IP_ADDRESS                = 1 << 8,
    QMI_WDS_REQUESTED_SETTINGS_GATEWAY_INFO              = 1 << 9,
    QMI_WDS_REQUESTED_SETTINGS_PCSCF_ADDRESS             = 1 << 10,
    QMI_WDS_REQUESTED_SETTINGS_PCSCF_SERVER_ADDRESS_LIST = 1 << 11,
    QMI_WDS_REQUESTED_SETTINGS_PCSCF_DOMAIN_NAME_LIST    = 1 << 12,
    QMI_WDS_REQUESTED_SETTINGS_MTU                       = 1 << 13,
    QMI_WDS_REQUESTED_SETTINGS_DOMAIN_NAME_LIST          = 1 << 14,
    QMI_WDS_REQUESTED_SETTINGS_IP_FAMILY                 = 1 << 15,
    QMI_WDS_REQUESTED_SETTINGS_IMCN_FLAG                 = 1 << 16,
    QMI_WDS_REQUESTED_SETTINGS_EXTENDED_TECHNOLOGY       = 1 << 17,
    QMI_WDS_REQUESTED_SETTINGS_OPERATOR_RESERVED_PCO     = 1 << 18,
} QmiWdsRequestedSettings;

/**
 * QmiWdsPdpType:
 * @QMI_WDS_PDP_TYPE_IPV4: IPv4
 * @QMI_WDS_PDP_TYPE_PPP: PPP
 * @QMI_WDS_PDP_TYPE_IPV6: IPv6
 * @QMI_WDS_PDP_TYPE_IPV4_OR_IPV6: IPv4 and IPv6 combined context
 *
 * PDP context type.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_PDP_TYPE_IPV4         = 0,
    QMI_WDS_PDP_TYPE_PPP          = 1,
    QMI_WDS_PDP_TYPE_IPV6         = 2,
    QMI_WDS_PDP_TYPE_IPV4_OR_IPV6 = 3
} QmiWdsPdpType;

/**
 * QmiWdsTrafficClass:
 * @QMI_WDS_TRAFFIC_CLASS_SUBSCRIBED: default class, best-effort
 * @QMI_WDS_TRAFFIC_CLASS_CONVERSATIONAL: two-way video/voice, most delay sensitive
 * @QMI_WDS_TRAFFIC_CLASS_STREAMING: one-way video/audio, delay sensitive
 * @QMI_WDS_TRAFFIC_CLASS_INTERACTIVE: delay-sensitive, e.g. browsing, SSH.
 * @QMI_WDS_TRAFFIC_CLASS_BACKGROUND: delay-insensitive, e.g. downloads, email.
 *
 * QoS Traffic Classes.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_TRAFFIC_CLASS_SUBSCRIBED     = 0,
    QMI_WDS_TRAFFIC_CLASS_CONVERSATIONAL = 1,
    QMI_WDS_TRAFFIC_CLASS_STREAMING      = 2,
    QMI_WDS_TRAFFIC_CLASS_INTERACTIVE    = 3,
    QMI_WDS_TRAFFIC_CLASS_BACKGROUND     = 4
} QmiWdsTrafficClass;

/**
 * QmiWdsAuthentication:
 * @QMI_WDS_AUTHENTICATION_NONE: no authentication
 * @QMI_WDS_AUTHENTICATION_PAP: PAP authentication
 * @QMI_WDS_AUTHENTICATION_CHAP: CHAP authentication
 *
 * PDP context authentication protocols.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_AUTHENTICATION_NONE = 0,
    QMI_WDS_AUTHENTICATION_PAP  = 1 << 0,
    QMI_WDS_AUTHENTICATION_CHAP = 1 << 1
} QmiWdsAuthentication;

/**
 * QmiWdsProfileType:
 * @QMI_WDS_PROFILE_TYPE_3GPP: 3GPP profile type.
 * @QMI_WDS_PROFILE_TYPE_3GPP2: 3GPP2 profile type.
 * @QMI_WDS_PROFILE_TYPE_EPC: EPC profile type. Since 1.34.
 * @QMI_WDS_PROFILE_TYPE_ALL: All technologies profile type. Since 1.34.
 *
 * Profile network type family.
 *
 * Since: 1.0
 */
typedef enum { /*< since=1.0 >*/
    QMI_WDS_PROFILE_TYPE_3GPP  = 0,
    QMI_WDS_PROFILE_TYPE_3GPP2 = 1,
    QMI_WDS_PROFILE_TYPE_EPC   = 2,
    QMI_WDS_PROFILE_TYPE_ALL   = 255
} QmiWdsProfileType;

 /**
  * QmiWdsProfileFamily:
  * @QMI_WDS_PROFILE_FAMILY_EMBEDDED: Embedded profile family.
  * @QMI_WDS_PROFILE_FAMILY_TETHERED: Tethered profile family.
  *
  * Profile family.
  *
  * Since: 1.22
  */
 typedef enum { /*< since=1.22 >*/
     QMI_WDS_PROFILE_FAMILY_EMBEDDED = 0,
     QMI_WDS_PROFILE_FAMILY_TETHERED = 1
 } QmiWdsProfileFamily;

/**
 * QmiWdsDeliveryOrder:
 * @QMI_WDS_QOS_DELIVERY_ORDER_SUBSCRIBE: Subscribe.
 * @QMI_WDS_QOS_DELIVERY_ORDER_ON: On.
 * @QMI_WDS_QOS_DELIVERY_ORDER_OFF: Off.
 *
 * QoS delivery order.
 *
 * Since: 1.14
 */
typedef enum { /*< since=1.14 >*/
    QMI_WDS_QOS_DELIVERY_ORDER_SUBSCRIBE = 0,
    QMI_WDS_QOS_DELIVERY_ORDER_ON        = 1,
    QMI_WDS_QOS_DELIVERY_ORDER_OFF       = 2
} QmiWdsDeliveryOrder;

/**
 * QmiWdsSduErrorRatio:
 * @QMI_WDS_SDU_ERROR_RATIO_SUBSCRIBE: Subscribe.
 * @QMI_WDS_SDU_ERROR_RATIO_1E2: 1x10^2.
 * @QMI_WDS_SDU_ERROR_RATIO_7E3: 7x10^3.
 * @QMI_WDS_SDU_ERROR_RATIO_1E3: 1x10^3.
 * @QMI_WDS_SDU_ERROR_RATIO_1E4: 1x10^4.
 * @QMI_WDS_SDU_ERROR_RATIO_1E5: 1x10^5.
 * @QMI_WDS_SDU_ERROR_RATIO_1E6: 1x10^6.
 * @QMI_WDS_SDU_ERROR_RATIO_1E1: 1x10^7.
 *
 * Target value for the fraction of SDUs lost or detected as erroneous.
 *
 * Since: 1.14
 */
typedef enum { /*< since=1.14 >*/
    QMI_WDS_SDU_ERROR_RATIO_SUBSCRIBE = 0,
    QMI_WDS_SDU_ERROR_RATIO_1E2       = 1,
    QMI_WDS_SDU_ERROR_RATIO_7E3       = 2,
    QMI_WDS_SDU_ERROR_RATIO_1E3       = 3,
    QMI_WDS_SDU_ERROR_RATIO_1E4       = 4,
    QMI_WDS_SDU_ERROR_RATIO_1E5       = 5,
    QMI_WDS_SDU_ERROR_RATIO_1E6       = 6,
    QMI_WDS_SDU_ERROR_RATIO_1E1       = 7
} QmiWdsSduErrorRatio;

/**
 * QmiWdsSduResidualBitErrorRatio:
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_SUBSCRIBE: Subscribe.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_5E2: 5x10^2.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E2: 1x10^2.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_5E3: 5x10^3.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_4E3: 4x10^3.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E3: 1x10^3.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E4: 1x10^4.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E5: 1x10^5.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E6: 1x10^6.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_6E8: 6x10^8.
 *
 * Target value for the undetected bit error ratio in the delivered SDUs.
 *
 * Since: 1.14
 */
typedef enum { /*< since=1.14 >*/
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_SUBSCRIBE = 0,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_5E2       = 1,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E2       = 2,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_5E3       = 3,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_4E3       = 4,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E3       = 5,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E4       = 6,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E5       = 7,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E6       = 8,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_6E8       = 9
} QmiWdsSduResidualBitErrorRatio;

/**
 * QmiWdsSduErroneousDelivery:
 * @QMI_WDS_SDU_ERRONEOUS_DELIVERY_SUBSCRIBE: Subscribe.
 * @QMI_WDS_SDU_ERRONEOUS_DELIVERY_NO_DETECTION: No detection.
 * @QMI_WDS_SDU_ERRONEOUS_DELIVERY_ENABLED: Erroneous SDUs are delivered.
 * @QMI_WDS_SDU_ERRONEOUS_DELIVERY_DISABLED: Erroneous SDUs are delivered.
 *
 * Whether SDUs detected as erroneous are delivered or not.
 *
 * Since: 1.14
 */
typedef enum { /*< since=1.14 >*/
    QMI_WDS_SDU_ERRONEOUS_DELIVERY_SUBSCRIBE    = 0,
    QMI_WDS_SDU_ERRONEOUS_DELIVERY_NO_DETECTION = 1,
    QMI_WDS_SDU_ERRONEOUS_DELIVERY_ENABLED      = 2,
    QMI_WDS_SDU_ERRONEOUS_DELIVERY_DISABLED     = 3
} QmiWdsSduErroneousDelivery;

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Packet Statistics' message */

/**
 * QmiWdsPacketStatisticsMaskFlag:
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_OK: Request count of correctly sent packets.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_OK: Request count of correctly received packets.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_ERROR: Request count of sent packets with error.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_ERROR: Request count of received packets with error.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_OVERFLOWS: Request count of transmitter overflows.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_OVERFLOWS: Request count of receiver overflows.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_BYTES_OK: Request count of correctly sent bytes.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_BYTES_OK: Request count of correctly received bytes.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_DROPPED: Request count of dropped packets in transmission.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_DROPPED: Request count of dropped packets in reception.
 *
 * Mask to use when requesting packet statistics.
 *
 * Since: 1.6
 */
typedef enum { /*< since=1.6 >*/
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_OK      = 1 << 0,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_OK      = 1 << 1,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_ERROR   = 1 << 2,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_ERROR   = 1 << 3,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_OVERFLOWS       = 1 << 4,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_OVERFLOWS       = 1 << 5,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_BYTES_OK        = 1 << 6,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_BYTES_OK        = 1 << 7,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_DROPPED = 1 << 8,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_DROPPED = 1 << 9
} QmiWdsPacketStatisticsMaskFlag;

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Profile List' message */

/**
 * QmiWdsDsProfileError:
 * @QMI_WDS_DS_PROFILE_ERROR_UNKNOWN: Unknown DS profile error.
 * @QMI_WDS_DS_PROFILE_ERROR_FAIL: General failure.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_HANDLE: Invalid handle.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_OPERATION: Invalid operation.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_TYPE: Invalid profile type.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_NUMBER: Invalid profile number.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_ID: Invalid profile identifier.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_ARGUMENT: Invalid argument.
 * @QMI_WDS_DS_PROFILE_ERROR_REGISTRY_NOT_INITIALIZED: Registry not initialized.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_PARAMETER_LENGTH: Invalid parameter length.
 * @QMI_WDS_DS_PROFILE_ERROR_LIST_END: List end, profile not found.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_SUBSCRIPTION_ID: Invalid subscription identifier.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_FAMILY: Invalid profile family.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_INVALID_PROFILE_FAMILY: Invalid 3GPP profile family.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_ACCESS_ERROR: Error accessing 3GPP profiles.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_CONTEXT_NOT_DEFINED: 3GPP profile doesn't have a valid context.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_VALID_FLAG_NOT_SET: 3GPP profile doesn't have valid flag set.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_READ_ONLY_FLAG_SET: 3GPP profile has read-only flag set.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_OUT_OF_PROFILES: Out of 3GPP profiles.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP2_INVALID_PROFILE_ID: Invalid 3GPP2 profile identifier.
 *
 * Extended DS profile errors.
 *
 * Since: 1.8
 */
typedef enum { /*< since=1.8 >*/
    QMI_WDS_DS_PROFILE_ERROR_UNKNOWN                     = 0,
    QMI_WDS_DS_PROFILE_ERROR_FAIL                        = 1,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_HANDLE              = 2,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_OPERATION           = 3,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_TYPE        = 4,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_NUMBER      = 5,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_ID          = 6,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_ARGUMENT            = 7,
    QMI_WDS_DS_PROFILE_ERROR_REGISTRY_NOT_INITIALIZED    = 8,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_PARAMETER_LENGTH    = 9,
    QMI_WDS_DS_PROFILE_ERROR_LIST_END                    = 10,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_SUBSCRIPTION_ID     = 11,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_FAMILY      = 12,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_INVALID_PROFILE_FAMILY = 1001,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_ACCESS_ERROR           = 1002,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_CONTEXT_NOT_DEFINED    = 1003,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_VALID_FLAG_NOT_SET     = 1004,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_READ_ONLY_FLAG_SET     = 1005,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_OUT_OF_PROFILES        = 1006,
    QMI_WDS_DS_PROFILE_ERROR_3GPP2_INVALID_PROFILE_ID    = 1101,
} QmiWdsDsProfileError;

/**
 * QmiWdsAutoconnectSetting:
 * @QMI_WDS_AUTOCONNECT_SETTING_DISABLED: Disabled.
 * @QMI_WDS_AUTOCONNECT_SETTING_ENABLED: Enabled.
 * @QMI_WDS_AUTOCONNECT_SETTING_PAUSED: Paused, resume on power cycle.
 *
 * Status of the autoconnect setting.
 *
 * Since: 1.14
 */
typedef enum { /*< since=1.14 >*/
    QMI_WDS_AUTOCONNECT_SETTING_DISABLED = 0,
    QMI_WDS_AUTOCONNECT_SETTING_ENABLED  = 1,
    QMI_WDS_AUTOCONNECT_SETTING_PAUSED   = 2,
} QmiWdsAutoconnectSetting;

/**
 * QmiWdsAutoconnectSettingRoaming:
 * @QMI_WDS_AUTOCONNECT_SETTING_ROAMING_ALLOWED: Autoconnect always allowed, even when roaming.
 * @QMI_WDS_AUTOCONNECT_SETTING_ROAMING_HOME_ONLY: Autoconnect allowed only in home network.
 *
 * Status of the autoconnect setting while roaming.
 *
 * Since: 1.14
 */
typedef enum { /*< since=1.14 >*/
    QMI_WDS_AUTOCONNECT_SETTING_ROAMING_ALLOWED   = 0,
    QMI_WDS_AUTOCONNECT_SETTING_ROAMING_HOME_ONLY = 1,
} QmiWdsAutoconnectSettingRoaming;

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Set Event Report' message */

/**
 * QmiWdsSetEventReportTransferStatistics:
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_OK: report TX packet successes.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_OK: report RX packet successes.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_ERROR: report TX packet errors.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_ERROR: report RX packet errors.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_OVERFLOWS: report TX overflows.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_OVERFLOWS: report RX overflows.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_BYTES_OK: report TX byte total.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_BYTES_OK: report RX byte total.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_DROPPED: report TX packets dropped.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_DROPPED: report RX packets dropped.
 *
 * Flags indicating which WDS Set Event Report Transfer Statistics indicators
 * should be emitted.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_OK      = 1 << 0,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_OK      = 1 << 1,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_ERROR   = 1 << 2,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_ERROR   = 1 << 3,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_OVERFLOWS       = 1 << 4,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_OVERFLOWS       = 1 << 5,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_BYTES_OK        = 1 << 6,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_BYTES_OK        = 1 << 7,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_DROPPED = 1 << 8,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_DROPPED = 1 << 9,
} QmiWdsSetEventReportTransferStatistics;

/**
 * QmiWdsDormancyStatus:
 * @QMI_WDS_DORMANCY_STATUS_UNKNOWN: dormancy status unknown.
 * @QMI_WDS_DORMANCY_STATUS_TRAFFIC_CHANNEL_DORMANT: traffic channel is dormant.
 * @QMI_WDS_DORMANCY_STATUS_TRAFFIC_CHANNEL_ACTIVE: traffic channel is active.
 *
 * Dormancy status.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_DORMANCY_STATUS_UNKNOWN                 = 0,
    QMI_WDS_DORMANCY_STATUS_TRAFFIC_CHANNEL_DORMANT = 1,
    QMI_WDS_DORMANCY_STATUS_TRAFFIC_CHANNEL_ACTIVE  = 2,
} QmiWdsDormancyStatus;

/**
 * QmiWdsDataCallStatus:
 * @QMI_WDS_DATA_CALL_STATUS_UNKNOWN: data call status unknown.
 * @QMI_WDS_DATA_CALL_STATUS_ACTIVATED: data call status activated.
 * @QMI_WDS_DATA_CALL_STATUS_TERMINATED: data call status terminated.
 *
 * Data call status.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_DATA_CALL_STATUS_UNKNOWN    = 0,
    QMI_WDS_DATA_CALL_STATUS_ACTIVATED  = 1,
    QMI_WDS_DATA_CALL_STATUS_TERMINATED = 2,
} QmiWdsDataCallStatus;

/**
 * QmiWdsDataCallType:
 * @QMI_WDS_DATA_CALL_TYPE_UNKNOWN: data call type unknown.
 * @QMI_WDS_DATA_CALL_TYPE_EMBEDDED: embedded data call.
 * @QMI_WDS_DATA_CALL_TYPE_TETHERED: tethered data call.
 * @QMI_WDS_DATA_CALL_TYPE_MODEM_EMBEDDED: modem embedded data call.
 *
 * Data call type.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_DATA_CALL_TYPE_UNKNOWN        = 0,
    QMI_WDS_DATA_CALL_TYPE_EMBEDDED       = 1,
    QMI_WDS_DATA_CALL_TYPE_TETHERED       = 2,
    QMI_WDS_DATA_CALL_TYPE_MODEM_EMBEDDED = 3,
} QmiWdsDataCallType;

/**
 * QmiWdsTetheredCallType:
 * @QMI_WDS_TETHERED_CALL_TYPE_NON_TETHERED: non-tethered.
 * @QMI_WDS_TETHERED_CALL_TYPE_RMNET: RmNet.
 * @QMI_WDS_TETHERED_CALL_TYPE_DUN: DUN.
 *
 * Tethered call type.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_TETHERED_CALL_TYPE_NON_TETHERED = 0,
    QMI_WDS_TETHERED_CALL_TYPE_RMNET        = 1,
    QMI_WDS_TETHERED_CALL_TYPE_DUN          = 2,
} QmiWdsTetheredCallType;

/**
 * QmiWdsRadioAccessTechnology:
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_NULL_BEARER: null bearer
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_WCDMA: WCDMA.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_GERAN: GERAN.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_LTE: LTE.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_TDSCDMA: TD-SDCMA.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_WLAN: 3GPP WLAN.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_MAXIMUM: 3GPP maximum.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_1X: CDMA 1x.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_HRPD: CDMA EVDO, HRPD.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_EHRPD: CDMA EVDO with eHRPD.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_WLAN: 3GPP2 WLAN.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_MAXIMUM: 3GPP2 maximum.
 *
 * Radio access technology.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_NULL_BEARER   = 0,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_WCDMA    = 1,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_GERAN    = 2,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_LTE      = 3,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_TDSCDMA  = 4,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_WLAN     = 5,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_MAXIMUM  = 100,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_1X      = 101,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_HRPD    = 102,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_EHRPD   = 103,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_WLAN    = 104,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_MAXIMUM = 200,
} QmiWdsRadioAccessTechnology;

/**
 * QmiWdsExtendedDataBearerTechnology3gpp:
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_UNKNOWN: unknown.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_WCDMA: WCDMA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSDPA: HSDPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSUPA: HSUPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSDPA_PLUS: HSDPA+.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_DC_HSDPA_PLUS: Dual-cell HSDPA+.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_64QAM: 64-QAM.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSPA: HSPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_GPRS: GPRS.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_EDGE: EDGE.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_GSM: GSM.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_S2B: S2B.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_LIMITED_SERVICE: LTE limited service.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_FDD: LTE FDD.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_TDD: LTE TDD.
 *
 * 3GPP extended data bearer technologies.
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_extended_data_bearer_technology_3gpp,since=1.18 >*/
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_UNKNOWN             = 0,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_WCDMA               = 1 << 0,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSDPA               = 1 << 1,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSUPA               = 1 << 2,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSDPA_PLUS          = 1 << 3,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_DC_HSDPA_PLUS       = 1 << 4,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_64QAM               = 1 << 5,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSPA                = 1 << 6,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_GPRS                = 1 << 7,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_EDGE                = 1 << 8,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_GSM                 = 1 << 9,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_S2B                 = 1 << 10,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_LIMITED_SERVICE = 1 << 11,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_FDD             = 1 << 12,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_TDD             = 1 << 13,
} QmiWdsExtendedDataBearerTechnology3gpp;

/**
 * QmiWdsExtendedDataBearerTechnology3gpp2:
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_UNKNOWN: unknown.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_RESERVED: reserved.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS95: CDMA1x IS-95.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS2000: CDMA1x IS-2000.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS2000_REL_A: CDMA1x IS-2000 Release A.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_0_DPA: HDR EVDO Rev 0 DPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_DPA: HDR EVDO Rev A DPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_DPA: HDR EVDO Rev B DPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_MPA: HDR EVDO Rev A MPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_MPA: HDR EVDO Rev B MPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_EMPA: HDR EVDO Rev A EMPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_EMPA: HDR EVDO Rev B EMPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_MMPA: HDR EVDO Rev B MMPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_EVDO_FMC: HDR EVDO FMC.
 *
 * 3GPP2 extended data bearer technologies.
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_extended_data_bearer_technology_3gpp2,since=1.18 >*/
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_UNKNOWN             = 0,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_RESERVED            = 1 << 0,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS95         = 1 << 1,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS2000       = 1 << 2,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS2000_REL_A = 1 << 3,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_0_DPA       = 1 << 4,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_DPA       = 1 << 5,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_DPA       = 1 << 6,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_MPA       = 1 << 7,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_MPA       = 1 << 8,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_EMPA      = 1 << 9,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_EMPA      = 1 << 10,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_MMPA      = 1 << 11,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_EVDO_FMC        = 1 << 12,
} QmiWdsExtendedDataBearerTechnology3gpp2;

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Profile'-related messages */

/**
 * QmiWdsPdpHeaderCompressionType:
 * @QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_OFF: no header compression
 * @QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_MANUFACTURER_PREFERRED: manufacturer preferred
 * @QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC1144: RFC1144
 * @QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC2507: RFC2507
 * @QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC3095: RFC3095
 *
 * PDP header compression types.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_OFF                    = 0,
    QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_MANUFACTURER_PREFERRED = 1,
    QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC1144                = 2,
    QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC2507                = 3,
    QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC3095                = 4,
} QmiWdsPdpHeaderCompressionType;

/**
 * QmiWdsPdpDataCompressionType:
 * @QMI_WDS_PDP_DATA_COMPRESSION_TYPE_OFF: no data compression
 * @QMI_WDS_PDP_DATA_COMPRESSION_TYPE_MANUFACTURER_PREFERRED: manufacturer preferred
 * @QMI_WDS_PDP_DATA_COMPRESSION_TYPE_V42BIS: V42bis
 * @QMI_WDS_PDP_DATA_COMPRESSION_TYPE_V44: V44
 *
 * PDP data compression types.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_PDP_DATA_COMPRESSION_TYPE_OFF                    = 0,
    QMI_WDS_PDP_DATA_COMPRESSION_TYPE_MANUFACTURER_PREFERRED = 1,
    QMI_WDS_PDP_DATA_COMPRESSION_TYPE_V42BIS                 = 2,
    QMI_WDS_PDP_DATA_COMPRESSION_TYPE_V44                    = 3,
} QmiWdsPdpDataCompressionType;

/**
 * QmiWdsQosClassIdentifier:
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_NETWORK_ASSIGN: network assigned
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE1: guaranteed bitrate 1
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE2: guaranteed bitrate 2
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE3: guaranteed bitrate 3
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE4: guaranteed bitrate 4
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE5: non-guaranteed bitrate 5
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE6: non-guaranteed bitrate 6
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE7: non-guaranteed bitrate 7
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE8: non-guaranteed bitrate 8
 *
 * QoS Class Identifier.
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_QOS_CLASS_IDENTIFIER_NETWORK_ASSIGN          = 0,
    QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE1     = 1,
    QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE2     = 2,
    QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE3     = 3,
    QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE4     = 4,
    QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE5 = 5,
    QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE6 = 6,
    QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE7 = 7,
    QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE8 = 8,
} QmiWdsQosClassIdentifier;

/**
 * QmiWdsAttachPdnListAction:
 * @QMI_WDS_ATTACH_PDN_LIST_ACTION_NONE: No action (default)
 * @QMI_WDS_ATTACH_PDN_LIST_ACTION_DETACH_OR_PDN_DISCONNECT: PS detach or PDN disconnect based on the modem configuration
 *
 * Action when setting the attach PDN list.
 *
 * Since: 1.28
 */
typedef enum { /*< since=1.28 >*/
    QMI_WDS_ATTACH_PDN_LIST_ACTION_NONE                     = 0x01,
    QMI_WDS_ATTACH_PDN_LIST_ACTION_DETACH_OR_PDN_DISCONNECT = 0x02,
} QmiWdsAttachPdnListAction;

/**
 * QmiWdsClientType:
 * @QMI_WDS_CLIENT_TYPE_TETHERED: client type tethered
 * @QMI_WDS_CLIENT_TYPE_UNDEFINED: no client type defined
 *
 * Client Type
 *
 * Since: 1.18
 */
typedef enum { /*< since=1.18 >*/
    QMI_WDS_CLIENT_TYPE_TETHERED = 0x01,
    QMI_WDS_CLIENT_TYPE_UNDEFINED = 0xFF,
} QmiWdsClientType;

/**
 * QMI_WDS_RATE_UNAVAILABLE:
 *
 * The TX/RX rate information is unavailable or cannot be measured.
 *
 * Since: 1.22.4
 */
#define QMI_WDS_RATE_UNAVAILABLE 0xFFFFFFFF

/**
 * QmiWdsIpSupportType:
 * @QMI_WDS_IP_SUPPORT_TYPE_IPV4: IPv4 type.
 * @QMI_WDS_IP_SUPPORT_TYPE_IPV6: IPv6 type.
 * @QMI_WDS_IP_SUPPORT_TYPE_IPV4V6: IPv4v6 type.
 *
 * IP support type.
 *
 * Since: 1.28
 */
typedef enum {  /*< since=1.28 >*/
    QMI_WDS_IP_SUPPORT_TYPE_IPV4   = 0,
    QMI_WDS_IP_SUPPORT_TYPE_IPV6   = 1,
    QMI_WDS_IP_SUPPORT_TYPE_IPV4V6 = 2,
} QmiWdsIpSupportType;

/**
 * QmiWdsProfileChangeEvent:
 * @QMI_WDS_PROFILE_CREATE_EVENT: create profile event.
 * @QMI_WDS_PROFILE_DELETE_EVENT: delete profile event.
 * @QMI_WDS_PROFILE_MODIFY_EVENT: modify profile event.
 * @QMI_WDS_PROFILE_SUBSCRIPTION_CHANGE_EVENT: subscription changed event.
 *
 * Profile change event.
 *
 * Since: 1.34
 */
typedef enum { /*< since=1.34 >*/
    QMI_WDS_PROFILE_CREATE_EVENT              = 1,
    QMI_WDS_PROFILE_DELETE_EVENT              = 2,
    QMI_WDS_PROFILE_MODIFY_EVENT              = 3,
    QMI_WDS_PROFILE_SUBSCRIPTION_CHANGE_EVENT = 4
} QmiWdsProfileChangeEvent;

#endif /* _LIBQMI_GLIB_QMI_ENUMS_WDS_H_ */
